<?php
// controllers/CompanyController.php
require_once 'models/CompanyModel.php';
require_once 'core/Auth.php';
require_once 'core/Session.php';

class CompanyController {
    private $companyModel;
    private $auth;
    private $session;
    
    public function __construct() {
        $this->companyModel = new CompanyModel();
        $this->auth = new Auth();
        $this->session = new Session();
        
        // التحقق من الصلاحيات
        if (!$this->auth->isLoggedIn()) {
            header('Location: login.php');
            exit;
        }
        
        if (!$this->auth->checkPermission('companies', 'read')) {
            $this->session->setFlash('ليس لديك صلاحية للوصول إلى هذه الصفحة', 'error');
            header('Location: index.php');
            exit;
        }
    }
    
    public function index() {
        $page = $_GET['page'] ?? 1;
        $perPage = 10;
        $search = $_GET['search'] ?? '';
        
        $companies = $this->companyModel->getAll($page, $perPage, $search);
        $totalCompanies = $this->companyModel->getTotalCount($search);
        $totalPages = ceil($totalCompanies / $perPage);
        
        $auth = $this->auth;
        $session = $this->session;
        include 'views/companies/index.php';
    }
    
    public function create() {
        if (!$this->auth->checkPermission('companies', 'create')) {
            $this->session->setFlash('ليس لديك صلاحية لإضافة شركات', 'error');
            header('Location: companies.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'phone' => trim($_POST['phone']),
                    'address' => trim($_POST['address']),
                    'opening_balance' => (float)$_POST['opening_balance']
                ];
                
                if (empty($data['name'])) {
                    throw new Exception('اسم الشركة مطلوب');
                }
                
                // ✅ فحص التكرار بالاسم فقط
                $existingCompany = $this->companyModel->getByName($data['name']);
                if ($existingCompany) {
                    $error = 'الشركة موجودة بالفعل بنفس الاسم!';
                    include 'views/companies/create.php';
                    return;
                }
                
                $this->companyModel->create($data);
                
                $this->session->setFlash('تم إضافة الشركة بنجاح', 'success');
                header('Location: companies.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        $auth = $this->auth;
        $session = $this->session;
        include 'views/companies/create.php';
    }
    
    public function edit($id) {
        if (!$this->auth->checkPermission('companies', 'update')) {
            $this->session->setFlash('ليس لديك صلاحية لتعديل الشركات', 'error');
            header('Location: companies.php');
            exit;
        }
        
        $company = $this->companyModel->getById($id);
        if (!$company) {
            $this->session->setFlash('الشركة غير موجودة', 'error');
            header('Location: companies.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'phone' => trim($_POST['phone']),
                    'address' => trim($_POST['address'])
                ];
                
                if (empty($data['name'])) {
                    throw new Exception('اسم الشركة مطلوب');
                }
                
                // ✅ فحص التكرار مع استثناء الشركة الحالية
                $existingCompany = $this->companyModel->getByName($data['name'], $id);
                if ($existingCompany) {
                    $error = 'يوجد شركة أخرى بنفس الاسم!';
                    include 'views/companies/edit.php';
                    return;
                }
                
                $this->companyModel->update($id, $data);
                
                $this->session->setFlash('تم تحديث الشركة بنجاح', 'success');
                header('Location: companies.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        $auth = $this->auth;
        $session = $this->session;
        include 'views/companies/edit.php';
    }
    
    public function delete($id) {
        if (!$this->auth->checkPermission('companies', 'delete')) {
            $this->session->setFlash('ليس لديك صلاحية لحذف الشركات', 'error');
            header('Location: companies.php');
            exit;
        }
        
        $company = $this->companyModel->getById($id);
        if (!$company) {
            $this->session->setFlash('الشركة غير موجودة', 'error');
        } else {
            $this->companyModel->delete($id);
            $this->session->setFlash('تم حذف الشركة بنجاح', 'success');
        }
        
        header('Location: companies.php');
        exit;
    }
}
