<?php
// controllers/CustomerController.php
require_once 'models/CustomerModel.php';
require_once 'core/Auth.php';
require_once 'core/Session.php';

class CustomerController {
    private $customerModel;
    private $auth;
    private $session;
    
    public function __construct() {
        $this->customerModel = new CustomerModel();
        $this->auth = new Auth();
        $this->session = new Session();
        
        // التحقق من الصلاحيات
        if (!$this->auth->isLoggedIn()) {
            header('Location: login.php');
            exit;
        }
        
        if (!$this->auth->checkPermission('customers', 'read')) {
            $this->session->setFlash('ليس لديك صلاحية للوصول إلى هذه الصفحة', 'error');
            header('Location: index.php');
            exit;
        }
    }
    
    public function index() {
        $page = $_GET['page'] ?? 1;
        $perPage = 10;
        $search = $_GET['search'] ?? '';
        
        $customers = $this->customerModel->getAll($page, $perPage, $search);
        $totalCustomers = $this->customerModel->getTotalCount($search);
        $totalPages = ceil($totalCustomers / $perPage);
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/customers/index.php';
    }
    
    public function create() {
        if (!$this->auth->checkPermission('customers', 'create')) {
            $this->session->setFlash('ليس لديك صلاحية لإضافة عملاء', 'error');
            header('Location: customers.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $name = trim($_POST['name']);
                $data = [
                    'name' => $name,
                    'phone' => trim($_POST['phone']),
                    'address' => trim($_POST['address']),
                    'opening_balance' => (float)$_POST['opening_balance']
                ];
                
                // التحقق من صحة البيانات
                if (empty($name)) {
                    throw new Exception('اسم العميل مطلوب');
                }
                
                // منع الإضافة إذا الاسم موجود مسبقًا
                $existingCustomer = $this->customerModel->getByName($name);
                if ($existingCustomer) {
                    throw new Exception('هذا الاسم موجود مسبقًا');
                }
                
                $customerId = $this->customerModel->create($data);
                
                $this->session->setFlash('تم إضافة العميل بنجاح', 'success');
                header('Location: customers.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/customers/create.php';
    }
    
    public function edit($id) {
        if (!$this->auth->checkPermission('customers', 'update')) {
            $this->session->setFlash('ليس لديك صلاحية لتعديل العملاء', 'error');
            header('Location: customers.php');
            exit;
        }
        
        $customer = $this->customerModel->getById($id);
        if (!$customer) {
            $this->session->setFlash('العميل غير موجود', 'error');
            header('Location: customers.php');
            exit;
        }
        
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $name = trim($_POST['name']);
                $data = [
                    'name' => $name,
                    'phone' => trim($_POST['phone']),
                    'address' => trim($_POST['address'])
                ];
                
                // التحقق من صحة البيانات
                if (empty($name)) {
                    throw new Exception('اسم العميل مطلوب');
                }

                // منع التعديل إذا الاسم موجود مسبقًا في سجل آخر
                $existingCustomer = $this->customerModel->getByName($name, $id);
                if ($existingCustomer) {
                    throw new Exception('هذا الاسم موجود مسبقًا');
                }
                
                $this->customerModel->update($id, $data);
                
                $this->session->setFlash('تم تحديث العميل بنجاح', 'success');
                header('Location: customers.php');
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        // تمرير المتغيرات للفيو
        $auth = $this->auth;
        $session = $this->session;
        include 'views/customers/edit.php';
    }
    
    public function delete($id) {
        if (!$this->auth->checkPermission('customers', 'delete')) {
            $this->session->setFlash('ليس لديك صلاحية لحذف العملاء', 'error');
            header('Location: customers.php');
            exit;
        }
        
        $customer = $this->customerModel->getById($id);
        if (!$customer) {
            $this->session->setFlash('العميل غير موجود', 'error');
        } else {
            $this->customerModel->delete($id);
            $this->session->setFlash('تم حذف العميل بنجاح', 'success');
        }
        
        header('Location: customers.php');
        exit;
    }
}
?>
