<?php
// controllers/ItemController.php
require_once 'models/ItemModel.php';
require_once 'models/UnitModel.php';
require_once 'models/CompanyModel.php';
require_once 'core/Auth.php';
require_once 'core/Session.php';

class ItemController {
    private $itemModel;
    private $unitModel;
    private $companyModel;
    private $auth;
    private $session;
    
    public function __construct() {
        $this->itemModel = new ItemModel();
        $this->unitModel = new UnitModel();
        $this->companyModel = new CompanyModel();
        $this->auth = new Auth();
        $this->session = new Session();
        
        // التحقق من الصلاحيات
        if (!$this->auth->isLoggedIn()) {
            echo "<script>alert('يجب تسجيل الدخول أولاً'); window.location.href='login.php';</script>";
            exit;
        }
        
        if (!$this->auth->checkPermission('items', 'read')) {
            echo "<script>alert('ليس لديك صلاحية للوصول إلى هذه الصفحة'); window.location.href='index.php';</script>";
            exit;
        }
    }
    
    public function index() {
        $page = $_GET['page'] ?? 1;
        $perPage = 10;
        $search = $_GET['search'] ?? '';
        
        $items = $this->itemModel->getAll($page, $perPage, $search);
        $totalItems = $this->itemModel->getTotalCount($search);
        $totalPages = ceil($totalItems / $perPage);
        
        $auth = $this->auth;
        $session = $this->session;
        include 'views/items/index.php';
    }
    
    public function create() {
        if (!$this->auth->checkPermission('items', 'create')) {
            echo "<script>alert('ليس لديك صلاحية لإضافة أصناف'); window.location.href='items.php';</script>";
            exit;
        }
        
        $units = $this->unitModel->getAllActive();
        $companies = $this->companyModel->getAllActive();
        $error = '';
        $success = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'default_price' => (float)$_POST['default_price'],
                    'model' => trim($_POST['model']),
                    'specifications' => trim($_POST['specifications']),
                    'unit_id' => (int)$_POST['unit_id'],
                    'company_id' => !empty($_POST['company_id']) ? (int)$_POST['company_id'] : null,
                    'current_stock' => (float)$_POST['current_stock']
                ];
                
                if (empty($data['name'])) {
                    throw new Exception('اسم الصنف مطلوب');
                }
                
                if ($data['default_price'] < 0) {
                    throw new Exception('السعر الافتراضي يجب أن يكون موجباً');
                }
                
                $existingItem = $this->itemModel->getByNameAndModel($data['name'], $data['model']);
                if ($existingItem) {
                    $error = 'الصنف موجود بالفعل بنفس الاسم والموديل!';
                    include 'views/items/create.php';
                    return;
                }
                
                if (!empty($_FILES['image']['name'])) {
                    $uploadResult = $this->handleImageUpload();
                    if ($uploadResult['success']) {
                        $data['image'] = $uploadResult['filename'];
                    } else {
                        throw new Exception($uploadResult['error']);
                    }
                }
                
                $itemId = $this->itemModel->create($data);
                $success = 'تم إضافة الصنف بنجاح';
                $_POST = [];
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        $auth = $this->auth;
        $session = $this->session;
        include 'views/items/create.php';
    }
    
    public function edit($id) {
        if (!$this->auth->checkPermission('items', 'update')) {
            echo "<script>alert('ليس لديك صلاحية لتعديل الأصناف'); window.location.href='items.php';</script>";
            exit;
        }
        
        $item = $this->itemModel->getById($id);
        if (!$item) {
            echo "<script>alert('الصنف غير موجود'); window.location.href='items.php';</script>";
            exit;
        }
        
        $units = $this->unitModel->getAllActive();
        $companies = $this->companyModel->getAllActive();
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $data = [
                    'name' => trim($_POST['name']),
                    'default_price' => (float)$_POST['default_price'],
                    'model' => trim($_POST['model']),
                    'specifications' => trim($_POST['specifications']),
                    'unit_id' => (int)$_POST['unit_id'],
                    'company_id' => !empty($_POST['company_id']) ? (int)$_POST['company_id'] : null,
                    'current_stock' => (float)$_POST['current_stock']
                ];
                
                if (empty($data['name'])) {
                    throw new Exception('اسم الصنف مطلوب');
                }
                
                if ($data['default_price'] < 0) {
                    throw new Exception('السعر الافتراضي يجب أن يكون موجباً');
                }
                
                $existingItem = $this->itemModel->getByNameAndModel($data['name'], $data['model'], $id);
                if ($existingItem) {
                    $error = 'الصنف موجود بالفعل بنفس الاسم والموديل!';
                    include 'views/items/edit.php';
                    return;
                }
                
                if (!empty($_FILES['image']['name'])) {
                    $uploadResult = $this->handleImageUpload();
                    if ($uploadResult['success']) {
                        $data['image'] = $uploadResult['filename'];
                        if (!empty($item['image'])) {
                            $this->deleteImage($item['image']);
                        }
                    } else {
                        throw new Exception($uploadResult['error']);
                    }
                }
                
                $this->itemModel->update($id, $data);
                echo "<script>alert('تم تحديث الصنف بنجاح'); window.location.href='items.php';</script>";
                exit;
                
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
        
        $auth = $this->auth;
        $session = $this->session;
        include 'views/items/edit.php';
    }
    
    public function delete($id) {
        if (!$this->auth->checkPermission('items', 'delete')) {
            echo "<script>alert('ليس لديك صلاحية لحذف الأصناف'); window.location.href='items.php';</script>";
            exit;
        }
        
        $item = $this->itemModel->getById($id);
        if (!$item) {
            echo "<script>alert('الصنف غير موجود');</script>";
        } else {
            if (!empty($item['image'])) {
                $this->deleteImage($item['image']);
            }
            
            $this->itemModel->delete($id);
            echo "<script>alert('تم حذف الصنف بنجاح');</script>";
        }
        
        echo "<script>window.location.href='items.php';</script>";
        exit;
    }
    
    private function handleImageUpload() {
        $uploadDir = 'uploads/items/';
        
        // إنشاء المجلد إذا لم يكن موجوداً
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        $file = $_FILES['image'];
        
        // التحقق من نوع الملف
        if (!in_array($file['type'], $allowedTypes)) {
            return ['success' => false, 'error' => 'نوع الملف غير مسموح به. المسموح: JPG, PNG, GIF'];
        }
        
        // التحقق من حجم الملف
        if ($file['size'] > $maxSize) {
            return ['success' => false, 'error' => 'حجم الملف كبير جداً. الحد الأقصى: 5MB'];
        }
        
        // إنشاء اسم فريد للملف
        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = uniqid() . '_' . time() . '.' . $extension;
        $filepath = $uploadDir . $filename;
        
        // رفع الملف
        if (move_uploaded_file($file['tmp_name'], $filepath)) {
            return ['success' => true, 'filename' => $filename];
        } else {
            return ['success' => false, 'error' => 'فشل في رفع الملف'];
        }
    }
    
    private function deleteImage($filename) {
        $filepath = 'uploads/items/' . $filename;
        if (file_exists($filepath)) {
            unlink($filepath);
        }
    }
}
?>