<?php
// models/ItemModel.php
require_once 'core/Database.php';

class ItemModel {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
 public function getAll($page = 1, $perPage = 10, $search = '') {
    $offset = ($page - 1) * $perPage;
    $where = [];
    $params = [];

    // شرط البحث
    if (!empty($search)) {
        $where[] = "(i.name LIKE :search OR i.model LIKE :search2)";
        $params[':search'] = "%$search%";
        $params[':search2'] = "%$search%";
    }

    // شرط التفعيل
    $where[] = "i.is_active = 1";
    $whereClause = "WHERE " . implode(" AND ", $where);

    // استعلام SQL
    $sql = "SELECT i.*, u.name as unit_name, c.name as company_name 
            FROM items i 
            LEFT JOIN units u ON i.unit_id = u.id 
            LEFT JOIN companies c ON i.company_id = c.id 
            {$whereClause}
            ORDER BY i.name ASC 
            LIMIT :limit OFFSET :offset";

    $stmt = $this->db->getConnection()->prepare($sql);

    // ربط معاملات البحث
    foreach ($params as $key => $val) {
        $stmt->bindValue($key, $val);
    }

    // ربط LIMIT و OFFSET كـ integers
    $stmt->bindValue(':limit', (int)$perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);

    $stmt->execute();

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

    
    public function getById($id) {
        $sql = "SELECT i.*, u.name as unit_name, c.name as company_name 
                FROM items i 
                LEFT JOIN units u ON i.unit_id = u.id 
                LEFT JOIN companies c ON i.company_id = c.id 
                WHERE i.id = ? AND i.is_active = 1";
        $stmt = $this->db->query($sql, [$id]);
        return $stmt->fetch();
    }
    
    public function create($data) {
        $this->db->beginTransaction();
        
        try {
            $sql = "INSERT INTO items (name, image, default_price, model, specifications, unit_id, company_id, current_stock, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())";
            
            $this->db->query($sql, [
                $data['name'],
                $data['image'] ?? null,
                $data['default_price'],
                $data['model'],
                $data['specifications'],
                $data['unit_id'],
                $data['company_id'] ?? null,
                $data['current_stock']
            ]);
            
            $itemId = $this->db->lastInsertId();
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'create', 'item', $itemId, null, $data);
            
            $this->db->commit();
            
            return $itemId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function update($id, $data) {
        $this->db->beginTransaction();
        
        try {
            $oldItem = $this->getById($id);
            
            $sql = "UPDATE items SET name = ?, default_price = ?, model = ?, specifications = ?, 
                    unit_id = ?, company_id = ?, current_stock = ?, updated_at = NOW()";
            $params = [
                $data['name'],
                $data['default_price'],
                $data['model'],
                $data['specifications'],
                $data['unit_id'],
                $data['company_id'] ?? null,
                $data['current_stock']
            ];
            
            // إذا كانت هناك صورة جديدة، أضفها إلى الاستعلام
            if (!empty($data['image'])) {
                $sql .= ", image = ?";
                $params[] = $data['image'];
            }
            
            $sql .= " WHERE id = ?";
            $params[] = $id;
            
            $this->db->query($sql, $params);
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'update', 'item', $id, $oldItem, $data);
            
            $this->db->commit();
            
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function delete($id) {
        $item = $this->getById($id);
        
        $sql = "UPDATE items SET is_active = 0, updated_at = NOW() WHERE id = ?";
        $result = $this->db->query($sql, [$id]);
        
        // تسجيل النشاط
        if ($result) {
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'delete', 'item', $id, $item);
        }
        
        return $result;
    }
    
    public function getTotalCount($search = '') {
        $where = [];
        $params = [];
        
        if (!empty($search)) {
            $where[] = "(name LIKE ? OR model LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        $where[] = "is_active = 1";
        $whereClause = "WHERE " . implode(" AND ", $where);
        
        $sql = "SELECT COUNT(*) as total FROM items {$whereClause}";
        $stmt = $this->db->query($sql, $params);
        return $stmt->fetch()['total'];
    }
    
    public function getAllActive() {
        $sql = "SELECT i.id, i.name, i.default_price, i.current_stock, u.name as unit_name 
                FROM items i 
                LEFT JOIN units u ON i.unit_id = u.id 
                WHERE i.is_active = 1 
                ORDER BY i.name";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }
    
    public function updateStock($itemId, $quantityChange) {
        $sql = "UPDATE items SET current_stock = current_stock + ?, updated_at = NOW() WHERE id = ?";
        return $this->db->query($sql, [$quantityChange, $itemId]);
    }
    
    private function logActivity($userId, $action, $entityType, $entityId, $oldValues = null, $newValues = null) {
        try {
            $sql = "INSERT INTO activity_logs 
                    (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            
            $this->db->query($sql, [
                $userId,
                $action,
                $entityType,
                $entityId,
                $oldValues ? json_encode($oldValues, JSON_UNESCAPED_UNICODE) : null,
                $newValues ? json_encode($newValues, JSON_UNESCAPED_UNICODE) : null,
                $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        } catch (Exception $e) {
            error_log("خطأ في تسجيل النشاط: " . $e->getMessage());
        }
    }
    public function getByNameAndModel($name, $model, $excludeId = null) {
    try {
        $sql = "SELECT * FROM items WHERE name = ? AND model = ? AND is_active = 1";
        $params = [$name, $model];
        
        if ($excludeId !== null) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }
        
        $stmt = $this->db->query($sql, $params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // يمكنك تسجيل الخطأ هنا إذا لزم الأمر
        return false;
    }
}
}
?>