<?php
require_once 'config.php';

class InvoiceManager {
    private $db;
    
    public function __construct() {
        $this->db = Config::getDB();
    }
    
    // إنشاء فاتورة جديدة
    public function createInvoice($data) {
        try {
            $this->db->beginTransaction();
            
            // توليد رقم فاتورة تلقائي
            $invoiceNumber = $this->generateInvoiceNumber();
            
            $stmt = $this->db->prepare("
                INSERT INTO invoices (
                    invoice_number, customer_id, currency_id, exchange_rate,
                    office_commission, shipping_costs, certificate_costs,
                    total_amount, total_amount_local, status, created_by
                ) VALUES (?, ?, ?, ?, ?, ?, ?, 0, 0, 'draft', ?)
            ");
            
            $stmt->execute([
                $invoiceNumber,
                $data['customer_id'],
                $data['currency_id'],
                $data['exchange_rate'],
                $data['office_commission'],
                $data['shipping_costs'],
                $data['certificate_costs'],
                $data['created_by']
            ]);
            
            $invoiceId = $this->db->lastInsertId();

             if (isset($data['exchange_rates']) && is_array($data['exchange_rates'])) {
        foreach ($data['exchange_rates'] as $currencyId => $rate) {
            $stmt = $this->db->prepare("INSERT INTO invoice_exchange_rates 
                                 (invoice_id, currency_id, exchange_rate, is_local) 
                                 VALUES (?, ?, ?, ?)");
            $stmt->execute([
                $invoiceId, 
                $currencyId, 
                $rate, 
                ($currencyId == $data['currency_id']) ? 1 : 0
            ]);
        }
    }
    
  
            $this->db->commit();
            
            return $invoiceId;
            
        } catch(Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    // إضافة صنف للفاتورة
    public function addInvoiceItem($invoiceId, $itemData) {
        try {
            $this->db->beginTransaction();
            
            // حساب السعر الإجمالي للصنف
            $totalPrice = $itemData['quantity'] * $itemData['unit_price'];
            
            $stmt = $this->db->prepare("
                INSERT INTO invoice_items (
                    invoice_id, item_id, company_id, quantity, 
                    unit_price, total_price
                ) VALUES (?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $invoiceId,
                $itemData['item_id'],
                $itemData['company_id'],
                $itemData['quantity'],
                $itemData['unit_price'],
                $totalPrice
            ]);
            
            $itemId = $this->db->lastInsertId();
            
            // تحديث إجمالي الفاتورة أولاً
            $this->updateInvoiceTotal($invoiceId);
            
            // إذا تم تحديد شركة موردة، نقوم بالتقييد المحاسبي
            if ($itemData['company_id']) {
                $this->recordCompanyTransaction($invoiceId, $itemData['company_id'], $totalPrice);
            }
            
            $this->db->commit();
            return $itemId;
            
        } catch(Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    // تحديث صنف موجود
    public function updateInvoiceItem($itemId, $itemData) {
        try {
            $this->db->beginTransaction();
            
            // الحصول على البيانات القديمة
            $oldItem = $this->getInvoiceItem($itemId);
            if (!$oldItem) {
                throw new Exception("الصنف غير موجود");
            }
            
            // حساب السعر الإجمالي الجديد
            $newTotalPrice = $itemData['quantity'] * $itemData['unit_price'];
            
            // تحديث الصنف
            $stmt = $this->db->prepare("
                UPDATE invoice_items 
                SET item_id = ?, company_id = ?, quantity = ?, 
                    unit_price = ?, total_price = ?
                WHERE id = ?
            ");
            
            $stmt->execute([
                $itemData['item_id'],
                $itemData['company_id'],
                $itemData['quantity'],
                $itemData['unit_price'],
                $newTotalPrice,
                $itemId
            ]);
            
            // إذا كانت هناك شركة موردة في البيانات القديمة، نعكس القيد
            if ($oldItem['company_id']) {
                $this->reverseCompanyTransaction($oldItem['invoice_id'], $oldItem['company_id'], $oldItem['total_price']);
            }
            
            // إذا كانت هناك شركة موردة في البيانات الجديدة، نقيم القيد
            if ($itemData['company_id']) {
                $this->recordCompanyTransaction($oldItem['invoice_id'], $itemData['company_id'], $newTotalPrice);
            }
            
            // تحديث إجمالي الفاتورة
            $this->updateInvoiceTotal($oldItem['invoice_id']);
            
            $this->db->commit();
            return true;
            
        } catch(Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    // حذف صنف من الفاتورة
    public function deleteInvoiceItem($itemId) {
        try {
            $this->db->beginTransaction();
            
            // الحصول على بيانات الصنف قبل الحذف
            $oldItem = $this->getInvoiceItem($itemId);
            if (!$oldItem) {
                throw new Exception("الصنف غير موجود");
            }
            
            $invoiceId = $oldItem['invoice_id'];
            
            // إذا كان للصنف شركة موردة، نعكس القيد المحاسبي
            if ($oldItem['company_id']) {
                $this->reverseCompanyTransaction($invoiceId, $oldItem['company_id'], $oldItem['total_price']);
            }
            
            // حذف الصنف
            $stmt = $this->db->prepare("DELETE FROM invoice_items WHERE id = ?");
            $stmt->execute([$itemId]);
            
            // تحديث إجمالي الفاتورة
            $this->updateInvoiceTotal($invoiceId);
            
            $this->db->commit();
            return true;
            
        } catch(Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    // تحديث تكاليف الفاتورة
    public function updateInvoiceCosts($invoiceId, $costsData) {
        try {
            $this->db->beginTransaction();
            
            $stmt = $this->db->prepare("
                UPDATE invoices 
                SET office_commission = ?, shipping_costs = ?, certificate_costs = ?,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            
            $stmt->execute([
                $costsData['office_commission'],
                $costsData['shipping_costs'],
                $costsData['certificate_costs'],
                $invoiceId
            ]);
            
            // تحديث إجمالي الفاتورة
            $this->updateInvoiceTotal($invoiceId);
            
            $this->db->commit();
            return true;
            
        } catch(Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    // الحصول على بيانات صنف معين
    // الحصول على بيانات صنف معين
public function getInvoiceItem($itemId) {
    $stmt = $this->db->prepare("
        SELECT ii.*, i.id as invoice_id, i.customer_id
        FROM invoice_items ii
        INNER JOIN invoices i ON ii.invoice_id = i.id
        WHERE ii.id = ?
    ");
    $stmt->execute([$itemId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
    // تحديث إجمالي الفاتورة والتقييد المحاسبي
    public function updateInvoiceTotal($invoiceId) {
        try {
            // حساب مجموع أصناف الفاتورة
            $stmt = $this->db->prepare("
                SELECT 
                    COALESCE(SUM(total_price), 0) as items_total,
                    i.exchange_rate,
                    i.office_commission,
                    i.shipping_costs,
                    i.certificate_costs,
                    i.customer_id,
                    i.currency_id
                FROM invoices i
                LEFT JOIN invoice_items ii ON i.id = ii.invoice_id
                WHERE i.id = ?
                GROUP BY i.id
            ");
            $stmt->execute([$invoiceId]);
            $invoiceData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$invoiceData) {
                throw new Exception("الفاتورة غير موجودة");
            }
            
            $itemsTotal = $invoiceData['items_total'];
            $exchangeRate = $invoiceData['exchange_rate'];
$officeCommissionPercent = $invoiceData['office_commission'];
$commissionValue = ($itemsTotal * $officeCommissionPercent) / 100;
            $shippingCosts = $invoiceData['shipping_costs'];
            $certificateCosts = $invoiceData['certificate_costs'];
            $customerId = $invoiceData['customer_id'];
            $currencyId = $invoiceData['currency_id'];
            
            // حساب الإجمالي بالعملة الأجنبية
            $totalAmount = $itemsTotal + $commissionValue + $shippingCosts + $certificateCosts;
            
            // حساب الإجمالي بالعملة المحلية
            $totalAmountLocal = $totalAmount * $exchangeRate;
            
            // تحديث الفاتورة بالمبالغ الجديدة
            $stmt = $this->db->prepare("
                UPDATE invoices 
                SET total_amount = ?, 
                    total_amount_local = ?,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$totalAmount, $totalAmountLocal, $invoiceId]);
            
            // تحديث التقييد المحاسبي للعميل
            $this->updateCustomerTransaction($invoiceId, $customerId, $totalAmountLocal, $currencyId, $exchangeRate);
            
            return [
                'items_total' => $itemsTotal,
                'total_amount' => $totalAmount,
                'total_amount_local' => $totalAmountLocal
            ];
            
        } catch(Exception $e) {
            throw new Exception("خطأ في تحديث إجمالي الفاتورة: " . $e->getMessage());
        }
    }
    
    // تحديث الحركة المالية للعميل
    private function updateCustomerTransaction($invoiceId, $customerId, $amount, $currencyId, $exchangeRate) {
        try {
            // حذف الحركات القديمة لهذه الفاتورة إذا وجدت
            $stmt = $this->db->prepare("
                DELETE FROM transactions 
                WHERE related_invoice_id = ? AND account_type = 'customer'
            ");
            $stmt->execute([$invoiceId]);
            
            // إذا كان المبلغ أكبر من صفر، نضيف حركة جديدة
            if ($amount > 0) {
                // الحصول على الرصيد الحالي للعميل
                $stmt = $this->db->prepare("SELECT current_balance FROM customers WHERE id = ?");
                $stmt->execute([$customerId]);
                $currentBalance = $stmt->fetchColumn();
                
                // حساب الرصيد الجديد
                $newBalance = $currentBalance + $amount;
                
                // إضافة الحركة المالية
                $stmt = $this->db->prepare("
                    INSERT INTO transactions (
                        date, account_type, account_id, related_invoice_id,
                        debit_amount, description, balance_after,
                        created_at
                    ) VALUES (CURDATE(), 'customer', ?, ?, ?, ?, ?, NOW())
                ");
                
                $description = "فاتورة مشتريات - رصيد مدين";
                $stmt->execute([
                    $customerId,
                    $invoiceId,
                    $amount,
                    $description,
                    $newBalance
                ]);
                
                // تحديث رصيد العميل
                $stmt = $this->db->prepare("
                    UPDATE customers 
                    SET current_balance = ?,
                        updated_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                ");
                $stmt->execute([$newBalance, $customerId]);
            }
            
        } catch(Exception $e) {
            throw new Exception("خطأ في تحديث الحركة المالية: " . $e->getMessage());
        }
    }
    
    // تسجيل حركة للشركة الموردة
    private function recordCompanyTransaction($invoiceId, $companyId, $amount) {
        try {
            // الحصول على الرصيد الحالي للشركة
            $stmt = $this->db->prepare("SELECT current_balance FROM companies WHERE id = ?");
            $stmt->execute([$companyId]);
            $currentBalance = $stmt->fetchColumn();
            
            // حساب الرصيد الجديد (دائن للشركة)
            $newBalance = $currentBalance + $amount;
            
            // إضافة الحركة المالية
            $stmt = $this->db->prepare("
                INSERT INTO transactions (
                    date, account_type, account_id, related_invoice_id,
                    credit_amount, description, balance_after,
                    created_at
                ) VALUES (CURDATE(), 'company', ?, ?, ?, ?, ?, NOW())
            ");
            
            $description = "مستحق من فاتورة مشتريات";
            $stmt->execute([
                $companyId,
                $invoiceId,
                $amount,
                $description,
                $newBalance
            ]);
            
            // تحديث رصيد الشركة
            $stmt = $this->db->prepare("
                UPDATE companies 
                SET current_balance = ?,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$newBalance, $companyId]);
            
        } catch(Exception $e) {
            throw new Exception("خطأ في تسجيل حركة الشركة: " . $e->getMessage());
        }
    }
    
    // عكس حركة الشركة الموردة
    private function reverseCompanyTransaction($invoiceId, $companyId, $amount) {
        try {
            // الحصول على الرصيد الحالي للشركة
            $stmt = $this->db->prepare("SELECT current_balance FROM companies WHERE id = ?");
            $stmt->execute([$companyId]);
            $currentBalance = $stmt->fetchColumn();
            
            // حساب الرصيد الجديد (عكس القيد)
            $newBalance = $currentBalance - $amount;
            
            // إضافة الحركة المالية المعاكسة
            $stmt = $this->db->prepare("
                INSERT INTO transactions (
                    date, account_type, account_id, related_invoice_id,
                    debit_amount, description, balance_after,
                    created_at
                ) VALUES (CURDATE(), 'company', ?, ?, ?, ?, ?, NOW())
            ");
            
            $description = "إلغاء مستحق من فاتورة مشتريات";
            $stmt->execute([
                $companyId,
                $invoiceId,
                $amount,
                $description,
                $newBalance
            ]);
            
            // تحديث رصيد الشركة
            $stmt = $this->db->prepare("
                UPDATE companies 
                SET current_balance = ?,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$newBalance, $companyId]);
            
        } catch(Exception $e) {
            throw new Exception("خطأ في عكس حركة الشركة: " . $e->getMessage());
        }
    }
    
    // تعيين قيم مشتركة لمجموعة من الأصناف
    public function assignSharedValue($itemIds, $type, $value) {
        try {
            $this->db->beginTransaction();
            
            // إنشاء سجل في الجدول المناسب
            switch($type) {
                case 'carton':
                    $table = 'shared_cartons';
                    $field = 'total_cartons';
                    $referenceField = 'carton_id';
                    break;
                case 'weight':
                    $table = 'shared_weights';
                    $field = 'total_weight';
                    $referenceField = 'weight_id';
                    break;
                case 'cbm':
                    $table = 'shared_cbm';
                    $field = 'total_cbm';
                    $referenceField = 'cbm_id';
                    break;
                default:
                    throw new Exception("نوع غير صحيح");
            }
            
            $stmt = $this->db->prepare("INSERT INTO $table ($field) VALUES (?)");
            $stmt->execute([$value]);
            $sharedId = $this->db->lastInsertId();
            
            // تحديث الأصناف المحددة
            $placeholders = str_repeat('?,', count($itemIds) - 1) . '?';
            $stmt = $this->db->prepare("
                UPDATE invoice_items 
                SET $referenceField = ? 
                WHERE id IN ($placeholders)
            ");
            
            $params = array_merge([$sharedId], $itemIds);
            $stmt->execute($params);
            
            $this->db->commit();
            return $sharedId;
            
        } catch(Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    // إلغاء القيم المشتركة
    public function removeSharedValue($itemIds, $type) {
        try {
            $this->db->beginTransaction();
            
            // تحديد الحقل المناسب
            switch($type) {
                case 'carton':
                    $referenceField = 'carton_id';
                    break;
                case 'weight':
                    $referenceField = 'weight_id';
                    break;
                case 'cbm':
                    $referenceField = 'cbm_id';
                    break;
                default:
                    throw new Exception("نوع غير صحيح");
            }
            
            // إلغاء القيم المشتركة
            $placeholders = str_repeat('?,', count($itemIds) - 1) . '?';
            $stmt = $this->db->prepare("
                UPDATE invoice_items 
                SET $referenceField = NULL 
                WHERE id IN ($placeholders)
            ");
            
            $stmt->execute($itemIds);
            
            $this->db->commit();
            return true;
            
        } catch(Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    // الحصول على الأصناف مجمعة حسب القيم المشتركة
    public function getGroupedInvoiceItems($invoiceId) {
        $stmt = $this->db->prepare("
            SELECT 
                ii.*, 
                it.name as item_name, 
                it.default_price, 
                u.name as unit_name, 
                co.name as company_name,
                sc.total_cartons, 
                sw.total_weight, 
                scbm.total_cbm,
                GROUP_CONCAT(DISTINCT ii.id) as grouped_item_ids,
                COUNT(*) OVER(PARTITION BY ii.carton_id) as carton_group_count,
                COUNT(*) OVER(PARTITION BY ii.weight_id) as weight_group_count,
                COUNT(*) OVER(PARTITION BY ii.cbm_id) as cbm_group_count
            FROM invoice_items ii
            LEFT JOIN items it ON ii.item_id = it.id
            LEFT JOIN units u ON it.unit_id = u.id
            LEFT JOIN companies co ON ii.company_id = co.id
            LEFT JOIN shared_cartons sc ON ii.carton_id = sc.id
            LEFT JOIN shared_weights sw ON ii.weight_id = sw.id
            LEFT JOIN shared_cbm scbm ON ii.cbm_id = scbm.id
            WHERE ii.invoice_id = ?
            ORDER BY 
                COALESCE(ii.carton_id, ii.weight_id, ii.cbm_id) DESC,
                ii.id ASC
        ");
        $stmt->execute([$invoiceId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // توليد رقم فاتورة تلقائي
    private function generateInvoiceNumber() {
        $year = date('Y');
        $month = date('m');
        $day = date('d');
        $hour = date('H')+3;
        $minute = date('i');
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as count 
            FROM invoices 
            WHERE YEAR(created_at) = ? AND MONTH(created_at) = ?
        ");
        $stmt->execute([$year, $month]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $sequence = $result['count'] + 1;
        return "GT-$year$month$day$hour$minute-" . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }
    
    // الحصول على تفاصيل الفاتورة
    public function getInvoice($invoiceId) {
        $stmt = $this->db->prepare("
            SELECT i.*, c.name as customer_name, cr.name as currency_name, 
                   cr.code as currency_code, u.username as created_by_name
            FROM invoices i
            LEFT JOIN customers c ON i.customer_id = c.id
            LEFT JOIN currencies cr ON i.currency_id = cr.id
            LEFT JOIN users u ON i.created_by = u.id
            WHERE i.id = ?
        ");
        $stmt->execute([$invoiceId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    // الحصول على أصناف الفاتورة
    public function getInvoiceItems($invoiceId) {
        $stmt = $this->db->prepare("
            SELECT ii.*, it.name as item_name, it.default_price, 
                   u.name as unit_name, co.name as company_name,
                   sc.total_cartons, sw.total_weight, scbm.total_cbm
            FROM invoice_items ii
            LEFT JOIN items it ON ii.item_id = it.id
            LEFT JOIN units u ON it.unit_id = u.id
            LEFT JOIN companies co ON ii.company_id = co.id
            LEFT JOIN shared_cartons sc ON ii.carton_id = sc.id
            LEFT JOIN shared_weights sw ON ii.weight_id = sw.id
            LEFT JOIN shared_cbm scbm ON ii.cbm_id = scbm.id
            WHERE ii.invoice_id = ?
            ORDER BY ii.id DESC
        ");
        $stmt->execute([$invoiceId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // الحصول على إجماليات الفاتورة
   public function getInvoiceTotals($invoiceId) {
    $stmt = $this->db->prepare("
        SELECT 
            COALESCE(SUM(ii.total_price), 0) as items_total,
            i.office_commission,
            (COALESCE(SUM(ii.total_price), 0) * i.office_commission / 100) AS commission_value,
            i.shipping_costs,
            i.certificate_costs,
            i.exchange_rate,
            (COALESCE(SUM(ii.total_price), 0) + (COALESCE(SUM(ii.total_price), 0) * i.office_commission / 100) + i.shipping_costs + i.certificate_costs) AS total_amount,
            i.total_amount_local,
            i.customer_id,
            i.currency_id
        FROM invoices i
        LEFT JOIN invoice_items ii ON i.id = ii.invoice_id
        WHERE i.id = ?
        GROUP BY i.id
    ");
    $stmt->execute([$invoiceId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

    
    // الحصول على حركات الفاتورة
    public function getInvoiceTransactions($invoiceId) {
        $stmt = $this->db->prepare("
            SELECT t.*, 
                   CASE 
                       WHEN t.account_type = 'customer' THEN c.name
                       WHEN t.account_type = 'company' THEN co.name
                       ELSE 'نقدي/بنك'
                   END as account_name
            FROM transactions t
            LEFT JOIN customers c ON t.account_type = 'customer' AND t.account_id = c.id
            LEFT JOIN companies co ON t.account_type = 'company' AND t.account_id = co.id
            WHERE t.related_invoice_id = ?
            ORDER BY t.created_at DESC
        ");
        $stmt->execute([$invoiceId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>