<?php
session_start();
require_once 'InvoiceManager.php';

$invoiceManager = new InvoiceManager();
$invoiceId = $_GET['invoice_id'] ?? 0;
$message = '';
$error = '';

// التحقق من معرف الفاتورة
if (!$invoiceId) {
    header("Location: invoices.php");
    exit;
}

// جلب بيانات الفاتورة
$invoice = $invoiceManager->getInvoice($invoiceId);
$items = $invoiceManager->getInvoiceItems($invoiceId);
$invoiceTotals = $invoiceManager->getInvoiceTotals($invoiceId);
$transactions = $invoiceManager->getInvoiceTransactions($invoiceId);

// جلب البيانات اللازمة
$db = Config::getDB();
$allItems = $db->query("
    SELECT i.*, u.name as unit_name, c.name as company_name 
    FROM items i 
    LEFT JOIN units u ON i.unit_id = u.id 
    LEFT JOIN companies c ON i.company_id = c.id 
    WHERE i.is_active = 1
")->fetchAll();

$companies = $db->query("SELECT id, name FROM companies WHERE is_active = 1")->fetchAll();

// جلب أسعار الصرف
$exchangeRates = [];
try {
    $stmt = $db->prepare("
        SELECT ier.*, c.code as currency_code, c.name as currency_name, c.symbol as currency_symbol
        FROM invoice_exchange_rates ier
        LEFT JOIN currencies c ON ier.currency_id = c.id
        WHERE ier.invoice_id = ?
        ORDER BY ier.is_local DESC, c.code ASC
    ");
    $stmt->execute([$invoiceId]);
    $exchangeRates = $stmt->fetchAll();
} catch(Exception $e) {
    // تجاهل الخطأ إذا لم يكن الجدول موجوداً بعد
}

// جلب العملات الأجنبية فقط
$foreignCurrencies = [];
try {
    $stmt = $db->prepare("
        SELECT c.id, c.code, c.name, c.symbol 
        FROM currencies c 
        WHERE c.is_active = 1 
        AND c.is_local = 0
        ORDER BY c.code ASC
    ");
    $stmt->execute();
    $foreignCurrencies = $stmt->fetchAll();
} catch(Exception $e) {
    // تجاهل الخطأ
}

// معالجة جميع الإجراءات في مكان واحد مع التحقق من التوكن
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $formToken = $_POST['form_token'] ?? '';
    
    // التحقق من التوكن
    if (!isset($_SESSION['form_token']) || $_SESSION['form_token'] !== $formToken) {
        $_SESSION['error'] = "توكن النموذج غير صالح";
        header("Location: invoice_items.php?invoice_id=" . $invoiceId);
        exit;
    }
    
    try {
        switch ($action) {
            case 'add_item':
                if (empty($_POST['item_id']) || empty($_POST['quantity']) || empty($_POST['unit_price'])) {
                    throw new Exception("جميع الحقول المطلوبة يجب ملؤها");
                }
                
                $invoiceManager->addInvoiceItem($invoiceId, [
                    'item_id' => $_POST['item_id'],
                    'company_id' => $_POST['company_id'] ?? null,
                    'quantity' => $_POST['quantity'],
                    'unit_price' => $_POST['unit_price']
                ]);
                
                $_SESSION['message'] = "تم إضافة الصنف بنجاح وتحديث الإجمالي";
                break;
                
            case 'update_invoice':
                $invoiceManager->updateInvoiceTotal($invoiceId);
                $_SESSION['message'] = "تم تحديث إجمالي الفاتورة بنجاح";
                break;
                
            case 'update_costs':
                $invoiceManager->updateInvoiceCosts($invoiceId, [
                    'office_commission' => $_POST['office_commission'] ?? 0,
                    'shipping_costs' => $_POST['shipping_costs'] ?? 0,
                    'certificate_costs' => $_POST['certificate_costs'] ?? 0
                ]);
                $_SESSION['message'] = "تم تحديث تكاليف الفاتورة بنجاح";
                break;
                
            case 'update_item':
                if (empty($_POST['item_id']) || empty($_POST['quantity']) || empty($_POST['unit_price'])) {
                    throw new Exception("جميع الحقول المطلوبة يجب ملؤها");
                }
                
                $invoiceManager->updateInvoiceItem($_POST['item_id'], [
                    'item_id' => $_POST['new_item_id'],
                    'company_id' => $_POST['company_id'] ?? null,
                    'quantity' => $_POST['quantity'],
                    'unit_price' => $_POST['unit_price']
                ]);
                
                $_SESSION['message'] = "تم تحديث الصنف بنجاح";
                break;
                
            case 'delete_item':
                if (empty($_POST['item_id'])) {
                    throw new Exception("معرف الصنف مطلوب");
                }
                
                $invoiceManager->deleteInvoiceItem($_POST['item_id']);
                $_SESSION['message'] = "تم حذف الصنف بنجاح";
                break;
                
            case 'assign_shared':
                $selectedItems = $_POST['selected_items'] ?? [];
                if (empty($selectedItems) || !isset($_POST['shared_value']) || $_POST['shared_value'] === '') {
                    throw new Exception("يرجى تحديد عناصر وإدخال قيمة");
                }
                
                $invoiceManager->assignSharedValue(
                    $selectedItems,
                    $_POST['shared_type'],
                    $_POST['shared_value']
                );
                $_SESSION['message'] = "تم تعيين القيمة المشتركة بنجاح";
                break;
                
            case 'remove_shared':
                $selectedItems = $_POST['selected_items'] ?? [];
                if (empty($selectedItems)) {
                    throw new Exception("يرجى تحديد عناصر");
                }
                
                $invoiceManager->removeSharedValue(
                    $selectedItems,
                    $_POST['shared_type']
                );
                $_SESSION['message'] = "تم إلغاء القيم المشتركة بنجاح";
                break;
                
            case 'update_exchange_container':
                // تحديث رقم الحاوية
                if (isset($_POST['container_number'])) {
                    $stmt = $db->prepare("UPDATE invoices SET container_number = ? WHERE id = ?");
                    $stmt->execute([$_POST['container_number'], $invoiceId]);
                }
                
                // تحديث أسعار الصرف الأجنبية
                if (isset($_POST['exchange_rates'])) {
                    $exchangeRatesData = $_POST['exchange_rates'];
                    
                    $db->beginTransaction();
                    
                    foreach ($exchangeRatesData as $currencyId => $rate) {
                        if (!empty($rate) && is_numeric($rate)) {
                            // التحقق إذا كان سعر الصرف موجوداً
                            $checkStmt = $db->prepare("
                                SELECT id FROM invoice_exchange_rates 
                                WHERE invoice_id = ? AND currency_id = ?
                            ");
                            $checkStmt->execute([$invoiceId, $currencyId]);
                            $exists = $checkStmt->fetch();
                            
                            if ($exists) {
                                $updateStmt = $db->prepare("
                                    UPDATE invoice_exchange_rates 
                                    SET exchange_rate = ?, created_at = NOW()
                                    WHERE invoice_id = ? AND currency_id = ?
                                ");
                                $updateStmt->execute([$rate, $invoiceId, $currencyId]);
                            } else {
                                $insertStmt = $db->prepare("
                                    INSERT INTO invoice_exchange_rates 
                                    (invoice_id, currency_id, exchange_rate, is_local, created_at) 
                                    VALUES (?, ?, ?, 0, NOW())
                                ");
                                $insertStmt->execute([$invoiceId, $currencyId, $rate]);
                            }
                        }
                    }
                    
                    $db->commit();
                }
                
                $_SESSION['message'] = "تم تحديث أسعار الصرف ورقم الحاوية بنجاح";
                break;
        }
        
        // إعادة التوجيه بعد المعالجة الناجحة
        header("Location: invoice_items.php?invoice_id=" . $invoiceId);
        exit;
        
    } catch(Exception $e) {
        $_SESSION['error'] = "خطأ في تنفيذ العملية: " . $e->getMessage();
        if ($db->inTransaction()) {
            $db->rollBack();
        }
        header("Location: invoice_items.php?invoice_id=" . $invoiceId);
        exit;
    }
}

// استرجاع الرسائل من الجلسة
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    unset($_SESSION['message']);
}
if (isset($_SESSION['error'])) {
    $error = $_SESSION['error'];
    unset($_SESSION['error']);
}

// إنشاء توكن جديد للنموذج
$form_token = bin2hex(random_bytes(32));
$_SESSION['form_token'] = $form_token;
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
   <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إدارة أصناف الفاتورة - <?= htmlspecialchars($invoice['invoice_number'] ?? '') ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet">
    <link rel="stylesheet" href="fonts/Almarai-local/fonts/fonts.css">
      <script src="jquery-3.6.0.min.js"></script>
     <link href="bootstrap.min.css" rel="stylesheet">
   <link href="select2.min.css" rel="stylesheet" />
   <script src="bootstrap.bundle.min.js"></script>
   <script src="select2.min.js"></script>
    <style>
        :root {
             --primary-color: #0E4262;
            --secondary-color: #F19E2C;
            --light-bg: #f8f9fa;
            --border-color: #dee2e6;
        }

        body {
            font-family: 'Almarai', sans-serif;
            background-color: #f5f7fa;
            min-height: 100vh;
        }

        .card-custom {
            border: none;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            margin: 5px auto;
            max-width: 1400px;
        }

        .card-header-custom {
            background: #2c3e50;
            color: white;
            padding: 15px 20px;
            border-bottom: 3px solid #1a252f;
            margin-bottom: 0;
            border-radius: 8px 8px 0 0;
            
    position: sticky;
    top: 0;
    z-index: 1100;
            
        }

        .card-body-custom {
            padding: 0;
        }

        .btn-custom {
            border: none;
            border-radius: 0;
            padding: 8px 16px;
            font-size: 14px;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            transition: background-color 0.3s;
        }

        .btn-primary-custom {
            background: #3498db;
            color: white;
        }

        .btn-primary-custom:hover {
            background: #2980b9;
        }

        .btn-success-custom {
            background: #2ecc71;
            color: white;
        }

        .btn-success-custom:hover {
            background: #27ae60;
        }

        .btn-warning-custom {
            background: #f39c12;
            color: white;
        }

        .btn-warning-custom:hover {
            background: #d68910;
        }

        .btn-danger-custom {
            background: #e74c3c;
            color: white;
        }

        .btn-danger-custom:hover {
            background: #c0392b;
        }

        .btn-secondary-custom {
            background: #95a5a6;
            color: white;
        }

        .btn-secondary-custom:hover {
            background: #7f8c8d;
        }

        .form-control-custom {
            border: 1px solid #bdc3c7;
            border-radius: 0;
            padding: 8px 12px;
            font-size: 14px;
            width: 100%;
        }

        .form-control-custom:focus {
            outline: none;
            border-color: #3498db;
            box-shadow: 0 0 0 2px rgba(52,152,219,0.2);
        }

        /* تنسيق الجدول */
       /* بداية من السطر 160 تقريباً - عدل تنسيقات الجدول */

.table-modern {
    border-radius: var(--border-radius);
    overflow: hidden;
    box-shadow: var(--box-shadow);
    margin-top: 2px !important;
    border: 1px solid #000 !important; /* حدود خارجية سوداء */
}

.table-modern thead th {
    background: var(--primary-color);
    color: white;
    border: 1px solid #000 !important; /* حدود سوداء للرأس */
    padding: 9px; /* زيادة البادنج */
    font-weight: 600;
    font-size: 14px;
    text-align: center;
    vertical-align: middle;
    border-bottom: 1px solid #000 !important; /* خط أسود أسفل الرأس */
}

.table-modern tbody td {
    border: 1px solid #000 !important; /* حدود سوداء للخلايا */
    padding: 10px;
    vertical-align: middle;
    transition: var(--transition);
       font-size: 14px;
}

.table-modern tbody tr {
    transition: var(--transition);
    border-bottom: 1px solid #000 !important; /* حدود أسفل كل صف */
}

.table-modern tbody tr:hover {
    background-color: rgba(67, 97, 238, 0.05);
}

.table-modern tfoot th {
    border: 1px solid #000 !important; /* حدود للتذييل */
    border-top: 2px solid #000 !important; /* خط أسود أعلى التذييل */
}

/* تنسيق خاص لأعمدة القيم المشتركة */
.table-modern tbody td.text-center.align-middle {
    border-left: 1px solid #000 !important;
    border-right: 1px solid #000 !important;
    min-height: 50px;
    height: 100%;
    vertical-align: middle;
}

/* تنسيق خاص للخلايا التي لها rowspan */
.table-modern td[rowspan] {
    background-color: rgba(14, 66, 98, 0.05);
    border-left: 2px solid #000 !important;
    border-right: 2px solid #000 !important;
}

/* تحسين تنسيق النصوص داخل الجدول */
.table-modern tbody td strong {
    color: var(--primary-color);
    font-weight: 600;
    font-size: 12px;
}

.table-modern tbody td .text-end {
    
    font-family: 'Courier New', monospace;
    font-weight: 600;
}

/* تنسيق الأزرار داخل الجدول */
.table-modern .action-buttons .btn-group-sm .btn {
    border: 1px solid #dee2e6;
    margin: 1px;
}

.table-modern .action-buttons .btn-outline-primary:hover {
    background-color: #0E4262;
    border-color: #0E4262;
}

.table-modern .action-buttons .btn-outline-danger:hover {
    background-color: #dc3545;
    border-color: #dc3545;
}

/* تنسيق حواف الجدول الخارجية */
.table-modern thead tr:first-child th:first-child {
    border-top-left-radius: 0;
}

.table-modern thead tr:first-child th:last-child {
    border-top-right-radius: 0;
}

.table-modern tbody tr:last-child td {
    border-bottom: 1px solid #000 !important;
}

/* تحسين مظهر checkboxes */
.table-modern .form-check-input {
    width: 18px;
    height: 18px;
    margin: 0 auto;
    border: 1px solid #495057;
}

.table-modern .form-check-input:checked {
    background-color: #0E4262;
    border-color: #0E4262;
}

/* تنسيق تذييل الجدول */
.table-modern tfoot {
    background-color: #2c3e50;
}

.table-modern tfoot th {
    color: white;
    font-weight: 600;
    padding: 10px;
}

.table-modern tfoot .text-success {
    color: #28a745 !important;
    font-size: 14px;
}

        /* تنسيق الإحصائيات */
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            padding: 10px;
            border-bottom: 1px solid #000;
        }

        .stats-card {
            background: white;
            padding: 15px;
            border-left: 4px solid;
            border-radius: 4px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .stats-card:nth-child(1) { border-color: #1abc9c; }
        .stats-card:nth-child(2) { border-color: #f1c40f; }
        .stats-card:nth-child(3) { border-color: #3498db; }
        .stats-card:nth-child(4) { border-color: #e74c3c; }

        .stats-number {
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .stats-label {
            font-size: 12px;
            color: #7f8c8d;
        }

        /* تنسيق الأقسام */
        .section-container {
            padding: 10px;
            border-bottom: 1px solid #000;
        }

        .section-title {
            font-weight: 600;
            font-size: 16px;
            color: #2c3e50;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
        }

        .section-title i {
            margin-left: 8px;
        }

        /* تنسيق الرسائل */
        .alert-custom {
            margin: 10px 20px;
            padding: 10px 15px;
            border-radius: 0;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .alert-success-custom {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-danger-custom {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        /* تنسيق الأزرار الجماعية */
        .bulk-actions {
            padding: 10px;
            margin-bottom: 1px;
        }

        /* تنسيق النماذج */
        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 10px;
        }

        /* تنسيق Select2 */
        .select2-container--default .select2-selection--single {
            border: 1px solid #bdc3c7 !important;
            border-radius: 0 !important;
            height: 38px !important;
            padding: 6px 10px !important;
        }

        .select2-container--default .select2-selection--single .select2-selection__rendered {
            color: #2c3e50 !important;
        }
         .select2-container--default .select2-results__option--highlighted[aria-selected] {
        background-color: #F19E2C !important;
        color: white !important;
    }

        @media (max-width: 768px) {
            .stats-container {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .table-custom {
                font-size: 12px;
            }
            
            .table-custom thead th,
            .table-custom tbody td {
                padding: 8px 6px;
            }
        }
        
        /* تنسيق جديد للأقسام المصفوفة */
        .row-section {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .section-box {
            background: white;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        
        .section-box h4 {
            margin-top: 0;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid #0E4262;
            color: #0E4262;
            font-weight: 600;
            font-size: 16px;
        }
    </style>
</head>
<body>
    <!-- رأس الصفحة -->
    <div class="card-custom">
        <div class="card-header-custom">
            <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap;">
                <div>
                    <h2 style="margin: 0; font-weight: 600; font-size: 18px;">
                        <i class="fas fa-file-invoice" style="margin-left: 8px;"></i>
                        إدارة أصناف الفاتورة
                    </h2>
                    <p style="margin: 5px 0 0 0; font-size: 14px; opacity: 0.9;">
                        رقم الفاتورة: <strong><?= htmlspecialchars($invoice['invoice_number'] ?? '') ?></strong> | 
                        العميل: <strong><?= htmlspecialchars($invoice['customer_name'] ?? '') ?></strong> |
                        العملة: <strong><?= htmlspecialchars($invoice['currency_code'] ?? '') ?></strong>
                    </p>
                </div>
                <div style="display: flex; gap: 10px;">
                    <a href="invoices.php" class="btn-custom btn-secondary-custom">
                        <i class="fas fa-arrow-right" style="margin-left: 8px;"></i>
                        العودة
                    </a>
                    <button type="button" class="btn-custom btn-primary-custom" onclick="location.reload();">
                        <i class="fas fa-sync-alt" style="margin-left: 8px;"></i>
                        تحديث
                    </button>
                </div>
            </div>
        </div>

        <div class="card-body-custom">
            <!-- رسائل النجاح والخطأ -->
            <?php if ($message): ?>
                <div class="alert-custom alert-success-custom">
                    <div style="display: flex; align-items: center;">
                        <i class="fas fa-check-circle" style="margin-left: 8px; color: #28a745;"></i>
                        <?= htmlspecialchars($message) ?>
                    </div>
                    <button onclick="this.parentElement.style.display='none'" style="background: none; border: none; color: #155724; cursor: pointer;">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert-custom alert-danger-custom">
                    <div style="display: flex; align-items: center;">
                        <i class="fas fa-exclamation-triangle" style="margin-left: 8px; color: #dc3545;"></i>
                        <?= htmlspecialchars($error) ?>
                    </div>
                    <button onclick="this.parentElement.style.display='none'" style="background: none; border: none; color: #721c24; cursor: pointer;">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            <?php endif; ?>

            <!-- الإحصائيات -->
            <div class="stats-container">
                <div class="stats-card">
                    <div class="stats-number"><?= count($items) ?></div>
                    <div class="stats-label">عدد الأصناف</div>
                </div>
                <div class="stats-card">
                    <div class="stats-number"><?= number_format($invoiceTotals['items_total'] ?? 0, 2) ?></div>
                    <div class="stats-label">مجموع الأصناف</div>
                </div>
                <div class="stats-card">
                    <div class="stats-number"><?= number_format($invoiceTotals['total_amount'] ?? 0, 2) ?></div>
                    <div class="stats-label">الإجمالي</div>
                </div>
                <div class="stats-card">
                    <div class="stats-number"><?= number_format($invoiceTotals['total_amount_local'] ?? 0, 2) ?></div>
                    <div class="stats-label">الإجمالي المحلي</div>
                </div>
            </div>

            <!-- إضافة صنف جديد -->
            <div class="section-container">
                <h3 class="section-title">
                    <i class="fas fa-plus-circle"></i>
                    إضافة صنف جديد
                </h3>
                <form method="POST" id="addItemForm">
                    <input type="hidden" name="action" value="add_item">
                    <input type="hidden" name="form_token" value="<?= $form_token ?>">
                    
                    <div class="form-grid">
                        <div>
                            <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">الصنف</label>
                            <select name="item_id" class="select2" id="itemSelect" required style="width: 100%;">
                                <option value="">اختر الصنف</option>
                                <?php foreach($allItems as $item): ?>
                                    <option value="<?= $item['id'] ?>" 
                                            data-price="<?= $item['default_price'] ?>"
                                            data-company="<?= $item['company_id'] ?>">
                                        <?= htmlspecialchars($item['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div>
                            <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">الكمية</label>
                            <input type="number" name="quantity" class="form-control-custom" value="1" step="0.001" required>
                        </div>
                        
                        <div>
                            <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">سعر الوحدة</label>
                            <input type="number" name="unit_price" id="unitPrice" class="form-control-custom" step="0.01" required>
                        </div>
                        
                        <div>
                            <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">الشركة</label>
                            <select name="company_id" class="select2" style="width: 100%;">
                                <option value="">اختر الشركة</option>
                                <?php foreach($companies as $company): ?>
                                    <option value="<?= $company['id'] ?>"><?= htmlspecialchars($company['name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div style="align-self: end;">
                            <button type="submit" class="btn-custom btn-success-custom" style="width: 100%;">
                                <i class="fas fa-plus" style="margin-left: 8px;"></i>
                                إضافة الصنف
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- أصناف الفاتورة -->
            <div class="section-container">
                <h3 class="section-title">
                    <i class="fas fa-list-ul"></i>
                    أصناف الفاتورة
                    <span style="background: #e8f4fd; padding: 2px 8px; border-radius: 2px; font-size: 12px; color: #3498db; margin-right: 10px;">
                        <?= count($items) ?> صنف
                    </span>
                </h3>

                <?php if (empty($items)): ?>
                    <div style="text-align: center; padding: 40px 20px; color: #7f8c8d;">
                        <i class="fas fa-inbox" style="font-size: 48px; margin-bottom: 15px; display: block; color: #ecf0f1;"></i>
                        لا توجد أصناف مضافة بعد
                    </div>
                <?php else: ?>
                    <!-- أزرار الإجراءات الجماعية -->
                    <div class="bulk-actions">
                        <div style="display: flex; gap: 10px; align-items: center; flex-wrap: wrap;">
                            <div style="display: flex; align-items: center;">
                                <input type="checkbox" id="selectAll" class="form-check-input" 
                                       style="margin-left: 8px; width: 18px; height: 18px;">
                                <label for="selectAll" style="font-size: 14px; color: #2c3e50;">تحديد الكل</label>
                            </div>
                            
                            <div style="flex: 1;">
                                <select id="sharedType" class="form-control-custom" style="font-size: 13px;">
                                    <option value="carton">الكراتين</option>
                                    <option value="weight">الوزن</option>
                                    <option value="cbm">CBM</option>
                                </select>
                            </div>
                            
                            <div style="min-width: 150px;">
                                <input type="number" id="sharedValue" class="form-control-custom" step="0.001" 
                                       placeholder="القيمة" style="font-size: 13px;">
                            </div>
                            
                            <div style="display: flex; gap: 5px;">
                                <button type="button" class="btn-custom btn-primary-custom" id="assignButton" disabled onclick="assignSharedValue()"
                                        style="padding: 5px 12px; font-size: 12px;">
                                    <i class="fas fa-link" style="margin-left: 5px; font-size: 11px;"></i>
                                    تعيين
                                </button>
                                
                                <button type="button" class="btn-custom btn-danger-custom" id="removeSharedButton" disabled onclick="removeSharedValue()"
                                        style="padding: 5px 12px; font-size: 12px;">
                                    <i class="fas fa-unlink" style="margin-left: 5px; font-size: 11px;"></i>
                                    إلغاء
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- جدول الأصناف -->
                    <?php
                    $carton_counts = [];
                    $weight_counts = [];
                    $cbm_counts = [];

                    foreach ($items as $item) {
                        if (!empty($item['carton_id'])) {
                            if (!isset($carton_counts[$item['carton_id']])) $carton_counts[$item['carton_id']] = 0;
                            $carton_counts[$item['carton_id']]++;
                        }
                        if (!empty($item['weight_id'])) {
                            if (!isset($weight_counts[$item['weight_id']])) $weight_counts[$item['weight_id']] = 0;
                            $weight_counts[$item['weight_id']]++;
                        }
                        if (!empty($item['cbm_id'])) {
                            if (!isset($cbm_counts[$item['cbm_id']])) $cbm_counts[$item['cbm_id']] = 0;
                            $cbm_counts[$item['cbm_id']]++;
                        }
                    }

                    $carton_printed = [];
                    $weight_printed = [];
                    $cbm_printed = [];
                    ?>
                    
                    <div style="overflow-x: auto;">
                       
<table class="table table-hover table-modern">
    <thead>
        <tr>
            <th width="50" class="text-center">#</th>
            <th>الصنف</th>
            <th class="text-center">الكمية</th>
            <th class="text-end">السعر</th>
            <th class="text-end">الإجمالي</th>
            <th class="text-center">الشركة</th>
            <th class="text-center">الكراتين</th>
            <th class="text-center">الوزن</th>
            <th class="text-center">CBM</th>
            <th width="120" class="text-center">الإجراءات</th>
        </tr>
    </thead>

    <tbody>
        <?php foreach ($items as $item): ?>
            <tr>
                <!-- اختيار -->
                <td class="text-center">
                    <input type="checkbox" name="selected_items[]" value="<?= $item['id'] ?>" class="form-check-input item-checkbox">
                </td>

                <!-- الصنف -->
                <td>
                    <strong><?= htmlspecialchars($item['item_name']) ?></strong>
                </td>

                <!-- الكمية -->
             <td class="text-center">
    <?= rtrim(rtrim((string)$item['quantity'], '0'), '.') ?>
</td>


                <!-- السعر -->
                <td class="text-end"><?= number_format($item['unit_price'], 2) ?></td>

                <!-- الإجمالي -->
                <td class="text-end"><?= number_format($item['total_price'], 2) ?></td>

                <!-- الشركة -->
                <td class="text-center"><?= $item['company_name'] ?? '-' ?></td>

                <!-- الكراتين -->
                <?php
                if (!empty($item['carton_id']) && empty($carton_printed[$item['carton_id']])) {
                    echo '<td class="text-center align-middle" rowspan="' . $carton_counts[$item['carton_id']] . '">'
                            . rtrim(rtrim((string)$item['total_cartons'], '0'), '.').
                        '</td>';
                    $carton_printed[$item['carton_id']] = true;
                } elseif (empty($item['carton_id'])) {
                           echo '            <td  class="text-center align-middle" style="text-align:center; font-size:14px;" >-</td>';
                }
                ?>

                <!-- الوزن -->
                <?php
                if (!empty($item['weight_id']) && empty($weight_printed[$item['weight_id']])) {
                    echo '<td class="text-center align-middle" rowspan="' . $weight_counts[$item['weight_id']] . '">'
                          . rtrim(rtrim((string)$item['total_weight'], '0'), '.').
                        '</td>';
                    $weight_printed[$item['weight_id']] = true;
                } elseif (empty($item['weight_id'])) {
                           echo '            <td  class="text-center align-middle" style="text-align:center; font-size:14px;" >-</td>';
                }
                ?>

                <!-- CBM -->
                <?php
                if (!empty($item['cbm_id']) && empty($cbm_printed[$item['cbm_id']])) {
                    echo '<td class="text-center align-middle" rowspan="' . $cbm_counts[$item['cbm_id']] . '">'
                        . rtrim(rtrim((string)$item['total_cbm'], '0'), '.') .
                        '</td>';
                    $cbm_printed[$item['cbm_id']] = true;
                } elseif (empty($item['cbm_id'])) {
                           echo '            <td  class="text-center align-middle" style="text-align:center; font-size:14px;" >-</td>';
                }
                ?>

                <!-- الإجراءات -->
                <td class="text-center action-buttons">
                    <div class="btn-group btn-group-sm">
                        <button type="button" class="btn btn-outline-primary" onclick="editItem(<?= $item['id'] ?>)">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button type="button" class="btn btn-outline-danger" onclick="deleteItem(<?= $item['id'] ?>)">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </td>

            </tr>
        <?php endforeach; ?>
    </tbody>

    <tfoot class="table-active">
        <tr>
            <th colspan="4" class="text-end" style="color: #0E4262;">المجموع النهائي + العمولة:</th>
            <th class="text-end text-success"><?= number_format($invoiceTotals['total_amount'] ?? 0, 2) ?></th>
            <th colspan="5"></th>
        </tr>
    </tfoot>
</table>
                    </div>
                <?php endif; ?>
            </div>

            <!-- قسم التكاليف وأسعار الصرف بشكل صفي -->
            <div class="section-container" style="border-bottom: none;">
                <div class="row-section">
                    <!-- تحديث تكاليف الفاتورة -->
                    <div class="section-box">
                        <h4>
                            <i class="fas fa-cog"></i>
                            تحديث تكاليف الفاتورة
                        </h4>
                        <form method="POST" id="updateCostsForm">
                            <input type="hidden" name="action" value="update_costs">
                            <input type="hidden" name="form_token" value="<?= $form_token ?>">
                            
                            <div style="margin-bottom: 15px;">
                                <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">عمولة المكتب</label>
                                <input type="number" name="office_commission" class="form-control-custom" 
                                       value="<?= $invoiceTotals['office_commission'] ?? 0 ?>" step="0.01">
                            </div>
                            <div style="margin-bottom: 15px;">
                                <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">تكاليف الشحن</label>
                                <input type="number" name="shipping_costs" class="form-control-custom" 
                                       value="<?= $invoiceTotals['shipping_costs'] ?? 0 ?>" step="0.01">
                            </div>
                            <div style="margin-bottom: 20px;">
                                <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">مصاريف الشهادات</label>
                                <input type="number" name="certificate_costs" class="form-control-custom" 
                                       value="<?= $invoiceTotals['certificate_costs'] ?? 0 ?>" step="0.01">
                            </div>
                            <button type="submit" class="btn-custom btn-success-custom" style="width: 100%; margin-top: 39px;">
                                <i class="fas fa-save" style="margin-left: 8px;"></i>
                                تحديث التكاليف
                            </button>
                        </form>
                    </div>

                    <!-- أسعار الصرف ورقم الحاوية -->
                    <div class="section-box">
                        <h4>
                            <i class="fas fa-money-bill-wave"></i>
                            أسعار الصرف ورقم الحاوية
                        </h4>
                        <form method="POST" id="updateExchangeContainerForm">
                            <input type="hidden" name="action" value="update_exchange_container">
                            <input type="hidden" name="form_token" value="<?= $form_token ?>">
                            
                            <div style="margin-bottom: 20px;">
                                <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">رقم الحاوية</label>
                                <input type="text" name="container_number" class="form-control-custom" 
                                       value="<?= htmlspecialchars($invoice['container_number'] ?? '') ?>" style="width: 100%;">
                            </div>
                            
                            <?php 
                            $foreignExchangeRates = array_filter($exchangeRates, function($rate) {
                                return $rate['is_local'] == 0;
                            });
                            
                            if (!empty($foreignExchangeRates)): ?>
                                <div style="background: #f8f9fa; padding: 15px; border: 1px solid #ecf0f1; margin-bottom: 20px;">
                                    <h5 style="margin: 0 0 10px 0; font-weight: 600; font-size: 12px; color: #2c3e50;">
                                        أسعار الصرف الأجنبية
                                    </h5>
                                    <div style="display: grid; grid-template-columns: 1fr; gap: 10px;">
                                        <?php foreach($foreignExchangeRates as $rate): ?>
                                            <div>
                                                <label style="display: block; margin-bottom: 5px; font-size: 11px; color: #7f8c8d;">
                                                    <?= $rate['currency_code'] ?> (<?= htmlspecialchars($rate['currency_name']) ?>)
                                                </label>
                                                <div style="display: flex; align-items: center;">
                                                    <span style="background: #e9ecef; padding: 6px 8px; border: 1px solid #bdc3c7; border-right: none; font-size: 11px; color: #7f8c8d; min-width: 80px;">
                                                        1 <?= $rate['currency_code'] ?> =
                                                    </span>
                                                    <input type="number" 
                                                           name="exchange_rates[<?= $rate['currency_id'] ?>]" 
                                                           class="form-control-custom" 
                                                           value="<?= number_format($rate['exchange_rate'], 6) ?>" 
                                                           step="0.000001"
                                                           placeholder="0.000000"
                                                           style="flex: 1; text-align: left; border-left: none; font-size: 12px; padding: 6px 8px;">
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                            
                            <button type="submit" class="btn-custom btn-success-custom" style="width: 100%;">
                                <i class="fas fa-save" style="margin-left: 8px;"></i>
                                تحديث اسعار الصرف 
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- نافذة تعديل الصنف -->
    <div id="editItemModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
        <div style="background: white; width: 90%; max-width: 500px; border-radius: 8px; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
            <div style="background: #2c3e50; color: white; padding: 15px 20px; border-radius: 8px 8px 0 0;">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                        <i class="fas fa-edit" style="margin-left: 8px;"></i>
                        تعديل الصنف
                    </h3>
                    <button onclick="hideEditModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
                </div>
            </div>
            <div style="padding: 20px;">
                <form id="editItemForm">
                    <input type="hidden" name="action" value="update_item">
                    <input type="hidden" name="form_token" value="<?= $form_token ?>">
                    <input type="hidden" name="item_id" id="editItemId">
                    
                    <div style="margin-bottom: 15px;">
                        <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">الصنف</label>
                        <select name="new_item_id" class="select2" id="editItemSelect" style="width: 100%;" required>
                            <option value="">اختر الصنف</option>
                            <?php foreach($allItems as $item): ?>
                                <option value="<?= $item['id'] ?>"><?= htmlspecialchars($item['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 20px;">
                        <div>
                            <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">الكمية</label>
                            <input type="number" name="quantity" id="editQuantity" class="form-control-custom" step="0.001" required>
                        </div>
                        <div>
                            <label style="display: block; margin-bottom: 5px; font-size: 12px; color: #7f8c8d;">سعر الوحدة</label>
                            <input type="number" name="unit_price" id="editUnitPrice" class="form-control-custom" step="0.01" required>
                        </div>
                    </div>
                </form>
            </div>
            <div style="padding: 15px 20px; border-top: 1px solid #ecf0f1; background: #f8f9fa; border-radius: 0 0 8px 8px; display: flex; justify-content: flex-end; gap: 10px;">
                <button onclick="hideEditModal()" class="btn-custom btn-secondary-custom">
                    إلغاء
                </button>
                <button type="button" onclick="submitEditForm()" class="btn-custom btn-primary-custom">
                    حفظ التعديلات
                </button>
            </div>
        </div>
    </div>

    <!-- نافذة تأكيد الحذف -->
    <div id="deleteModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
        <div style="background: white; width: 90%; max-width: 400px; border-radius: 8px; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
            <div style="background: #2c3e50; color: white; padding: 15px 20px; border-radius: 8px 8px 0 0;">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                        <i class="fas fa-trash" style="margin-left: 8px;"></i>
                        تأكيد الحذف
                    </h3>
                    <button onclick="hideDeleteModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
                </div>
            </div>
            <div style="padding: 20px;">
                <div style="text-align: center; margin-bottom: 20px;">
                    <i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #e74c3c; margin-bottom: 15px;"></i>
                    <p style="font-size: 16px; color: #2c3e50; margin-bottom: 10px;">هل أنت متأكد من حذف هذا الصنف؟</p>
                    <p style="font-size: 14px; color: #7f8c8d;">
                        سيتم حذف الصنف من الفاتورة ولا يمكن التراجع عن هذا الإجراء.
                    </p>
                </div>
                <div style="display: flex; gap: 10px; justify-content: flex-end;">
                    <button onclick="hideDeleteModal()" class="btn-custom btn-secondary-custom">
                        إلغاء
                    </button>
                    <button type="button" id="confirmDeleteBtn" class="btn-custom btn-danger-custom">
                        نعم، احذف
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
        // تهيئة Select2
         $(document).ready(function() {
    $('.select2').select2({
        dir: 'rtl',
        width: '100%'
    }).on('select2:open', function () {
        setTimeout(function () {
            var field = document.querySelector('.select2-container--open .select2-search__field');
            if (field) field.focus();
        }, 0);
    });

   $('#itemSelect').on('change', function () {
    var selected = $(this).find(':selected');

    // السعر
    $('#unitPrice').val(selected.data('price') || '');

    var companyId = selected.data('company');
    var companySelect = $('select[name="company_id"]');

    // هل الشركة موجودة ضمن الخيارات؟
    if (companyId && companySelect.find('option[value="' + companyId + '"]').length > 0) {
        companySelect.val(companyId).trigger('change');
    } else {
        // الشركة غير موجودة أو محذوفة → اختر شركة
        companySelect.val('').trigger('change');
    }
});

});

        // متغيرات عامة
        let currentItemId = null;

        // إدارة تحديد العناصر
        $('#selectAll').change(function() {
            $('.item-checkbox').prop('checked', this.checked).trigger('change');
        });

        $('.item-checkbox').change(function() {
            updateActionButtons();
        });

        function updateActionButtons() {
            var selectedCount = $('.item-checkbox:checked').length;
            $('#assignButton, #removeSharedButton').prop('disabled', selectedCount === 0);
        }

        function assignSharedValue() {
            var selectedItems = $('.item-checkbox:checked').map(function() {
                return this.value;
            }).get();
            var sharedValue = $('#sharedValue').val();
            var sharedType = $('#sharedType').val();
            
            if (selectedItems.length === 0 || !sharedValue) {
                alert('يرجى تحديد عناصر وإدخال قيمة');
                return;
            }

            var formData = new FormData();
            formData.append('action', 'assign_shared');
            formData.append('shared_type', sharedType);
            formData.append('shared_value', sharedValue);
            formData.append('form_token', '<?= $form_token ?>');
            selectedItems.forEach(itemId => {
                formData.append('selected_items[]', itemId);
            });

            fetch('invoice_items.php?invoice_id=<?= $invoiceId ?>', {
                method: 'POST',
                body: formData
            }).then(response => response.ok ? location.reload() : alert('حدث خطأ'));
        }

        function removeSharedValue() {
            var selectedItems = $('.item-checkbox:checked').map(function() {
                return this.value;
            }).get();
            var sharedType = $('#sharedType').val();
            
            if (selectedItems.length === 0) {
                alert('يرجى تحديد عناصر');
                return;
            }

            if (!confirm('هل أنت متأكد؟')) return;

            var formData = new FormData();
            formData.append('action', 'remove_shared');
            formData.append('shared_type', sharedType);
            formData.append('form_token', '<?= $form_token ?>');
            selectedItems.forEach(itemId => {
                formData.append('selected_items[]', itemId);
            });

            fetch('invoice_items.php?invoice_id=<?= $invoiceId ?>', {
                method: 'POST',
                body: formData
            }).then(response => response.ok ? location.reload() : alert('حدث خطأ'));
        }

        function editItem(itemId) {
            currentItemId = itemId;
            $('#editItemModal').css('display', 'flex');
        }

        function deleteItem(itemId) {
            currentItemId = itemId;
            $('#deleteModal').css('display', 'flex');
        }

        function hideEditModal() {
            $('#editItemModal').css('display', 'none');
        }

        function hideDeleteModal() {
            $('#deleteModal').css('display', 'none');
        }

        function submitEditForm() {
            var formData = new FormData(document.getElementById('editItemForm'));
            formData.append('item_id', currentItemId);
            
            fetch('invoice_items.php?invoice_id=<?= $invoiceId ?>', {
                method: 'POST',
                body: formData
            }).then(response => response.ok ? location.reload() : alert('حدث خطأ'));
        }

        // تأكيد الحذف
        $('#confirmDeleteBtn').click(function() {
            var formData = new FormData();
            formData.append('action', 'delete_item');
            formData.append('item_id', currentItemId);
            formData.append('form_token', '<?= $form_token ?>');
            
            fetch('invoice_items.php?invoice_id=<?= $invoiceId ?>', {
                method: 'POST',
                body: formData
            }).then(response => response.ok ? location.reload() : alert('حدث خطأ'));
        });

        // إغلاق النوافذ عند النقر خارجها
        window.onclick = function(event) {
            if (event.target == document.getElementById('editItemModal')) {
                hideEditModal();
            }
            if (event.target == document.getElementById('deleteModal')) {
                hideDeleteModal();
            }
        };
        $(document).ready(function () {
    $('#editItemSelect').select2({
        width: '100%',
        dropdownParent: $('#editItemModal')
    });
});
    </script>
</body>
</html>