<?php
// models/CompanyModel.php
require_once 'core/Database.php';

class CompanyModel {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = Database::getInstance(); // نستخدم singleton
        $this->conn = $this->db->getConnection(); // اتصال PDO
    }

    public function getByName($name, $excludeId = null) {
        $sql = "SELECT * FROM companies WHERE name = ?";
        $params = [$name];

        if ($excludeId !== null) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }

        $stmt = $this->conn->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
     
    public function getAll($page = 1, $perPage = 10, $search = '') {
    $offset = ($page - 1) * $perPage;
    $where = [];
    $params = [];

    if (!empty($search)) {
        $where[] = "(name LIKE :search OR phone LIKE :search2)";
        $params[':search'] = "%$search%";
        $params[':search2'] = "%$search%";
    }

    $where[] = "is_active = 1";
    $whereClause = "WHERE " . implode(" AND ", $where);

    $sql = "SELECT * FROM companies
            {$whereClause}
            ORDER BY name ASC
            LIMIT :limit OFFSET :offset";

    $stmt = $this->db->getConnection()->prepare($sql);

    // ربط معاملات البحث
    foreach ($params as $key => $val) {
        $stmt->bindValue($key, $val);
    }

    // ربط LIMIT و OFFSET
    $stmt->bindValue(':limit', (int)$perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);

    $stmt->execute();

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

    
    public function getById($id) {
        $sql = "SELECT * FROM companies WHERE id = ? AND is_active = 1";
        $stmt = $this->db->query($sql, [$id]);
        return $stmt->fetch();
    }
    
    public function create($data) {
        $this->db->beginTransaction();
        
        try {
            $sql = "INSERT INTO companies (name, phone, address, opening_balance, current_balance, created_at) 
                    VALUES (?, ?, ?, ?, ?, NOW())";
            
            $this->db->query($sql, [
                $data['name'],
                $data['phone'],
                $data['address'],
                $data['opening_balance'],
                $data['opening_balance'] // الرصيد الحالي يساوي الافتتاحي في البداية
            ]);
            
            $companyId = $this->db->lastInsertId();
            
            // تسجيل الحركة المالية للرصيد الافتتاحي
            if ($data['opening_balance'] > 0) {
                $this->recordOpeningBalanceTransaction($companyId, $data['opening_balance']);
            }
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'create', 'company', $companyId, null, $data);
            
            $this->db->commit();
            
            return $companyId;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function update($id, $data) {
        $this->db->beginTransaction();
        
        try {
            $oldCompany = $this->getById($id);
            
            $sql = "UPDATE companies SET name = ?, phone = ?, address = ?, updated_at = NOW() 
                    WHERE id = ?";
            
            $this->db->query($sql, [
                $data['name'],
                $data['phone'],
                $data['address'],
                $id
            ]);
            
            // تسجيل النشاط
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'update', 'company', $id, $oldCompany, $data);
            
            $this->db->commit();
            
            return true;
        } catch (Exception $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
    
    public function delete($id) {
        $company = $this->getById($id);
        
        $sql = "UPDATE companies SET is_active = 0, updated_at = NOW() WHERE id = ?";
        $result = $this->db->query($sql, [$id]);
        
        // تسجيل النشاط
        if ($result) {
            $this->logActivity($_SESSION['user']['id'] ?? 1, 'delete', 'company', $id, $company);
        }
        
        return $result;
    }
    
    public function getTotalCount($search = '') {
        $where = [];
        $params = [];
        
        if (!empty($search)) {
            $where[] = "(name LIKE ? OR phone LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }
        
        $where[] = "is_active = 1";
        $whereClause = "WHERE " . implode(" AND ", $where);
        
        $sql = "SELECT COUNT(*) as total FROM companies {$whereClause}";
        $stmt = $this->db->query($sql, $params);
        return $stmt->fetch()['total'];
    }
    
    public function getAllActive() {
        $sql = "SELECT id, name FROM companies WHERE is_active = 1 ORDER BY name";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }
    
    private function recordOpeningBalanceTransaction($companyId, $amount) {
        $sql = "INSERT INTO transactions 
                (date, account_type, account_id, debit_amount, credit_amount, description, balance_after, created_at) 
                VALUES (CURDATE(), 'company', ?, 0, ?, 'رصيد افتتاحي', ?, NOW())";
        
        $this->db->query($sql, [
            $companyId,
            $amount,
            -$amount // رصيد دائن للشركات
        ]);
    }
    
    private function logActivity($userId, $action, $entityType, $entityId, $oldValues = null, $newValues = null) {
        try {
            $sql = "INSERT INTO activity_logs 
                    (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            
            $this->db->query($sql, [
                $userId,
                $action,
                $entityType,
                $entityId,
                $oldValues ? json_encode($oldValues, JSON_UNESCAPED_UNICODE) : null,
                $newValues ? json_encode($newValues, JSON_UNESCAPED_UNICODE) : null,
                $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        } catch (Exception $e) {
            error_log("خطأ في تسجيل النشاط: " . $e->getMessage());
        }
    }
    
}


?>