<?php
// bank_accounts.php
$page_title = "إدارة الحسابات البنكية";
require_once 'config.php';

// بدء الجلسة إذا لم تكن بدأت
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// عرض رسائل الجلسة المؤقتة
if(isset($_SESSION['success_message'])) {
    $message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
} else {
    $message = '';
}

$action = $_POST['action'] ?? '';
$edit_id = $_GET['edit'] ?? 0;
$delete_id = $_GET['delete'] ?? 0;

// منع إعادة الإرسال (POST-Redirect-GET Pattern)
if($action === 'create_bank_account'){
    $name = sanitize($_POST['name']);
    $bank_name = sanitize($_POST['bank_name']);
    $account_number = sanitize($_POST['account_number']);
    $branch = sanitize($_POST['branch_name']);
    $iban = sanitize($_POST['iban']);
    $swift = sanitize($_POST['swift_code']);
    $opening = (float)$_POST['opening_balance'];
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    // التحقق من عدم تكرار رقم الحساب
    $check_sql = "SELECT id FROM bank_accounts WHERE account_number = ?";
    $check_stmt = $mysqli->prepare($check_sql);
    $check_stmt->bind_param('s', $account_number);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if($check_result->num_rows > 0) {
        $message = "رقم الحساب هذا مسجل مسبقاً!";
    } else {
        $sql = "INSERT INTO bank_accounts (name, bank_name, account_number, branch_name, iban, swift_code, opening_balance, current_balance, is_active, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $mysqli->prepare($sql);
        $t = date('Y-m-d H:i:s');
        $stmt->bind_param('ssssssddis', $name, $bank_name, $account_number, $branch, $iban, $swift, $opening, $opening, $is_active, $t);
        
        if($stmt->execute()){
            $bank_id = $stmt->insert_id;
            
            if($opening != 0){
                $desc = "رصيد افتتاحي للحساب البنكي: $name";
                $newbal = insert_transaction($mysqli, date('Y-m-d'), 'bank', 'receipt', $bank_id, null, null, $opening, 0.00, $desc, 'bank');
                update_container_balance($mysqli, 'bank_accounts', $bank_id, $newbal);
            }
            
            log_activity($mysqli, current_user_id(), 'create', 'bank_account', $bank_id, null, json_encode(['name'=>$name,'opening_balance'=>$opening]));
            
            // استخدام الجلسة لتخزين الرسالة وإعادة التوجيه
            $_SESSION['success_message'] = 'تم إنشاء الحساب البنكي بنجاح.';
            header("Location: bank_accounts.php");
            exit();
        } else {
            $message = 'حدث خطأ أثناء حفظ الحساب: ' . $stmt->error;
        }
        $stmt->close();
    }
    $check_stmt->close();
}

if($action === 'update_bank_account'){
    $id = (int)$_POST['id'];
    $name = sanitize($_POST['name']);
    $bank_name = sanitize($_POST['bank_name']);
    $account_number = sanitize($_POST['account_number']);
    $branch = sanitize($_POST['branch_name']);
    $iban = sanitize($_POST['iban']);
    $swift = sanitize($_POST['swift_code']);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    // التحقق من عدم تكرار رقم الحساب (باستثناء الحساب الحالي)
    $check_sql = "SELECT id FROM bank_accounts WHERE account_number = ? AND id != ?";
    $check_stmt = $mysqli->prepare($check_sql);
    $check_stmt->bind_param('si', $account_number, $id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if($check_result->num_rows > 0) {
        $message = "رقم الحساب هذا مسجل مسبقاً لحساب آخر!";
    } else {
        $res = $mysqli->query("SELECT name, bank_name, account_number, branch_name, opening_balance, current_balance, is_active FROM bank_accounts WHERE id=$id LIMIT 1");
        $old = $res->fetch_assoc();
        
        $sql = "UPDATE bank_accounts SET name=?, bank_name=?, account_number=?, branch_name=?, iban=?, swift_code=?, is_active=?, updated_at=? WHERE id=?";
        $stmt = $mysqli->prepare($sql);
        $t = date('Y-m-d H:i:s');
        $stmt->bind_param('ssssssisi', $name, $bank_name, $account_number, $branch, $iban, $swift, $is_active, $t, $id);
        
        if($stmt->execute()){
            log_activity($mysqli, current_user_id(), 'update', 'bank_account', $id, json_encode($old), json_encode(['name'=>$name,'is_active'=>$is_active]));
            
            $_SESSION['success_message'] = 'تم تحديث بيانات الحساب البنكي بنجاح.';
            header("Location: bank_accounts.php");
            exit();
        } else {
            $message = 'حدث خطأ أثناء تحديث الحساب: ' . $stmt->error;
        }
        $stmt->close();
    }
    $check_stmt->close();
}

// معالجة حذف الحساب
if($delete_id > 0) {
    // التحقق من عدم وجود معاملات مرتبطة بالحساب
    $check_transactions = $mysqli->query("SELECT COUNT(*) as count FROM transactions WHERE container_type = 'bank' AND container_id = $delete_id");
    $result = $check_transactions->fetch_assoc();
    
    if($result['count'] > 0) {
        $message = "لا يمكن حذف الحساب لأنه مرتبط بمعاملات مالية!";
    } else {
        $mysqli->query("DELETE FROM bank_accounts WHERE id = $delete_id");
        log_activity($mysqli, current_user_id(), 'delete', 'bank_account', $delete_id, null, null);
        
        $_SESSION['success_message'] = 'تم حذف الحساب البنكي بنجاح.';
        header("Location: bank_accounts.php");
        exit();
    }
}

// جلب بيانات الحساب للتعديل
$editing_account = null;
if($edit_id > 0) {
    $result = $mysqli->query("SELECT * FROM bank_accounts WHERE id = $edit_id");
    if($result->num_rows > 0) {
        $editing_account = $result->fetch_assoc();
    }
}

require_once 'header.php';
?>

<div class="card" style="border: none; box-shadow: 0 2px 10px rgba(0,0,0,0.08); border-radius: 0;padding: 8px;  margin: 5px auto; max-width: 1200px;">
 <div class="card-header" style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f; margin-bottom: 0;  position: sticky; top: 0;
z-index: 1100;">
        <div style="display: flex; justify-content: space-between; align-items: center;">
            <h2 style="margin: 0; font-weight: 600; font-size: 18px;">
                <i class="fas fa-university" style="margin-left: 8px;"></i>
                إدارة الحسابات البنكية
            </h2>
            <button type="button" class="btn" onclick="showAddForm()" 
                   style="padding: 8px 16px; background: #2ecc71; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center; transition: background 0.3s;"
                   onmouseover="this.style.background='#27ae60'"
                   onmouseout="this.style.background='#2ecc71'">
                <i class="fas fa-plus" style="margin-left: 8px;"></i>
                إضافة حساب بنكي جديد
            </button>
        </div>
    </div>

    <div class="card-body" style="padding: 0;">
        <!-- شريط البحث والإحصائيات -->
        <div style="padding: 20px; border-bottom: 1px solid #ecf0f1;">
            <!-- شريط البحث -->
            <form method="GET" action="" style="margin-bottom: 10px;">
                <div style="display: flex; gap: 10px;">
                    <div style="position: relative; flex: 1;">
                        <input type="text" name="search" class="form-control" 
                               placeholder="ابحث باسم الحساب أو البنك أو رقم الحساب..." 
                               value="<?php echo htmlspecialchars($search ?? ''); ?>"
                               style="width: 100%; padding: 10px 12px 10px 40px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;">
                        <i class="fas fa-search" style="position: absolute; left: 12px; top: 50%; transform: translateY(-50%); color: #7f8c8d;"></i>
                    </div>
                    <button type="submit" class="btn" 
                            style="padding: 8px 20px; background: #3498db; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; display: inline-flex; align-items: center;">
                        <i class="fas fa-search" style="margin-left: 8px;"></i>
                        بحث
                    </button>
                    <?php if (!empty($search)): ?>
                        <a href="bank_accounts.php" class="btn" 
                           style="padding: 8px 16px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center;">
                            <i class="fas fa-times" style="margin-left: 8px;"></i>
                            إلغاء البحث
                        </a>
                    <?php endif; ?>
                </div>
            </form>

            <!-- الإحصائيات -->
            <?php
            // حساب الإحصائيات
            $result = $mysqli->query("SELECT 
                COUNT(*) as total_accounts,
                SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_accounts,
                SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive_accounts,
                SUM(current_balance) as total_balance
                FROM bank_accounts");
            $stats = $result->fetch_assoc();
            ?>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
                <div style="background: #e8f6f3; padding: 15px; border-left: 4px solid #1abc9c;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">إجمالي الحسابات</div>
                    <div style="font-size: 24px; font-weight: bold; color: #16a085;"><?php echo number_format($stats['total_accounts'] ?? 0); ?></div>
                </div>
                <div style="background: #fef9e7; padding: 15px; border-left: 4px solid #f1c40f;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">حسابات مفعلة</div>
                    <div style="font-size: 24px; font-weight: bold; color: #f39c12;"><?php echo number_format($stats['active_accounts'] ?? 0); ?></div>
                </div>
                <div style="background: #fdedec; padding: 15px; border-left: 4px solid #e74c3c;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">حسابات غير مفعلة</div>
                    <div style="font-size: 24px; font-weight: bold; color: #c0392b;"><?php echo number_format($stats['inactive_accounts'] ?? 0); ?></div>
                </div>
                <div style="background: #e8f4fd; padding: 15px; border-left: 4px solid #3498db;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">إجمالي الأرصدة</div>
                    <div style="font-size: 24px; font-weight: bold; color: #2980b9;">
                        <?php echo number_format($stats['total_balance'] ?? 0, 2); ?> 
                        <span style="font-size: 14px;">¥</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- رسائل النجاح والخطأ -->
        <?php if ($message): ?>
            <div style="margin: 10px 12px; padding: 8px 12px; background: #d4edda; color: #155724; border: 1px solid #c3e6cb; border-radius: 0; display: flex; align-items: center; justify-content: space-between;">
                <div style="display: flex; align-items: center;">
                    <i class="fas fa-check-circle" style="margin-left: 8px; color: #28a745;"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <button onclick="this.parentElement.style.display='none'" style="background: none; border: none; color: #155724; cursor: pointer;">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        <?php endif; ?>

        <!-- حاوية أزرار الإجراءات (ستظهر عند التحديد) -->
        <div id="actionsContainer" style="max-height: 0; overflow: hidden; transition: max-height 0.3s ease; margin: 0 20px;">
            <div style="display: flex; gap: 8px; padding: 15px; background: #f8f9fa; border-bottom: 1px solid #ecf0f1; flex-wrap: wrap;">
                <a href="javascript:void(0)" class="btn" id="editBtn"
                   style="padding: 6px 12px; background: #3498db; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center; transition: background 0.3s;"
                   onmouseover="this.style.background='#2980b9'"
                   onmouseout="this.style.background='#3498db'">
                    <i class="fas fa-edit" style="margin-left: 5px; font-size: 11px;"></i>
                    تعديل
                </a>

                <button type="button" class="btn" id="viewBtn"
                        style="padding: 6px 12px; background: #17a2b8; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#138496'"
                        onmouseout="this.style.background='#17a2b8'">
                    <i class="fas fa-eye" style="margin-left: 5px; font-size: 11px;"></i>
                    عرض التفاصيل
                </button>

                <button type="button" class="btn" id="transactionsBtn"
                        style="padding: 6px 12px; background: #9b59b6; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#8e44ad'"
                        onmouseout="this.style.background='#9b59b6'">
                    <i class="fas fa-exchange-alt" style="margin-left: 5px; font-size: 11px;"></i>
                    المعاملات
                </button>

                <button type="button" class="btn" id="toggleStatusBtn"
                        style="padding: 6px 12px; background: #28a745; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#218838'"
                        onmouseout="this.style.background='#28a745'">
                    <i class="fas fa-power-off" style="margin-left: 5px; font-size: 11px;"></i>
                    <span id="toggleStatusText">تفعيل</span>
                </button>

                <button type="button" class="btn" id="deleteBtn"
                        style="padding: 6px 12px; background: #e74c3c; color: white; border: none; border-radius: 0; font-size: 12px; cursor: pointer; display: inline-flex; align-items: center; transition: background 0.3s;"
                        onmouseover="this.style.background='#c0392b'"
                        onmouseout="this.style.background='#e74c3c'">
                    <i class="fas fa-trash" style="margin-left: 5px; font-size: 11px;"></i>
                    حذف
                </button>
            </div>
        </div>

        <!-- جدول الحسابات البنكية -->
        <div style="overflow-x: auto;">
            <table class="table" style="width: 100%; border-collapse: collapse;">
                <thead>
                    <tr style="background: #f8f9fa;">
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">#</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: right; font-weight: 600; color: #F19E2C; font-size: 14px;">اسم الحساب</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: right; font-weight: 600; color: #F19E2C; font-size: 14px;">البنك</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">رقم الحساب</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">الرصيد الحالي</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">IBAN</th>
                        <th style="padding: 12px; border-bottom: 1px solid #000; text-align: center; font-weight: 600; color: #F19E2C; font-size: 14px;">الحالة</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $bank_accounts = get_bank_accounts($mysqli);
                    $total_balance = 0;
                    $counter = 0;
                    
                    if($bank_accounts->num_rows > 0): 
                        while($ba = $bank_accounts->fetch_assoc()): 
                            $counter++;
                            $total_balance += $ba['current_balance'];
                    ?>
                    <tr style="border-bottom: 1px solid #ecf0f1; transition: background 0.3s; cursor: pointer;" 
                        onmouseover="this.style.backgroundColor='#f8f9fa'" 
                        onmouseout="this.style.backgroundColor='white'"
                        onclick="selectAccount(<?php echo $ba['id']; ?>, this)"
                        data-account-id="<?php echo $ba['id']; ?>"
                        data-account-name="<?php echo htmlspecialchars($ba['name']); ?>"
                        data-account-number="<?php echo htmlspecialchars($ba['account_number']); ?>"
                        data-account-status="<?php echo $ba['is_active']; ?>"
                        data-account-balance="<?php echo $ba['current_balance']; ?>"
                        data-bank-name="<?php echo htmlspecialchars($ba['bank_name']); ?>"
                        data-branch="<?php echo htmlspecialchars($ba['branch_name']); ?>"
                        data-iban="<?php echo htmlspecialchars($ba['iban']); ?>"
                        data-swift="<?php echo htmlspecialchars($ba['swift_code']); ?>">
                        
                        <td style="padding: 12px; text-align: center; color: #7f8c8d; font-size: 14px; vertical-align: middle;">
                            <?php echo $counter; ?>
                        </td>
                        
                        <td style="padding: 12px; vertical-align: middle;">
                            <div style="display: flex; align-items: center; justify-content: flex-end;">
                                <div style="text-align: right;">
                                    <strong style="color: #2c3e50; font-size: 14px;">
                                        <?php echo htmlspecialchars($ba['name']); ?>
                                    </strong>
                                    <div style="font-size: 12px; color: #7f8c8d; margin-top: 3px;">
                                        <?php echo htmlspecialchars($ba['branch_name']); ?>
                                    </div>
                                </div>
                            </div>
                        </td>
                        
                        <td style="padding: 12px; text-align: right; vertical-align: middle;">
                            <div style="color: #2c3e50; font-weight: 500; font-size: 14px;">
                                <?php echo htmlspecialchars($ba['bank_name']); ?>
                            </div>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <code style="background: #e8f4fd; padding: 4px 8px; border-radius: 2px; font-size: 13px; color: #3498db; font-weight: 600;">
                                <?php echo htmlspecialchars($ba['account_number']); ?>
                            </code>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <div style="font-weight: 600; color: #27ae60; font-size: 14px;">
                                <?php echo number_format($ba['current_balance'], 2); ?> 
                                <span style="font-size: 12px; color: #7f8c8d;">¥</span>
                            </div>
                            <?php if($ba['opening_balance'] != $ba['current_balance']): ?>
                                <div style="font-size: 11px; color: #95a5a6;">
                                    افتتاحي: <?php echo number_format($ba['opening_balance'], 2); ?> ¥
                                </div>
                            <?php endif; ?>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <?php if(!empty($ba['iban'])): ?>
                                <code style="background: #e8f6f3; padding: 4px 8px; border-radius: 2px; font-size: 12px; color: #16a085;">
                                    <?php echo htmlspecialchars($ba['iban']); ?>
                                </code>
                            <?php else: ?>
                                <span style="color: #95a5a6; font-size: 12px;">-</span>
                            <?php endif; ?>
                        </td>
                        
                        <td style="padding: 12px; text-align: center; vertical-align: middle;">
                            <?php if($ba['is_active']): ?>
                                <span style="background: #2ecc71; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">
                                    <i class="fas fa-check-circle" style="margin-left: 5px;"></i>
                                    مفعل
                                </span>
                            <?php else: ?>
                                <span style="background: #e74c3c; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">
                                    <i class="fas fa-times-circle" style="margin-left: 5px;"></i>
                                    غير مفعل
                                </span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="7" style="padding: 20px; text-align: center; color: #7f8c8d; font-size: 14px;">
                                <i class="fas fa-university" style="font-size: 48px; color: #ecf0f1; margin-bottom: 15px; display: block;"></i>
                                لا توجد حسابات بنكية
                                <?php if (!empty($search)): ?>
                                    <div style="margin-top: 10px; font-size: 13px;">
                                        لم يتم العثور على نتائج لبحثك: "<?php echo htmlspecialchars($search ?? ''); ?>"
                                    </div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <!-- الترويج والتذييل -->
        <div style="padding: 20px; border-top: 1px solid #ecf0f1; background: #f8f9fa;">
            <?php if ($counter > 0): ?>
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div style="font-size: 13px; color: #7f8c8d;">
                    <i class="fas fa-info-circle" style="margin-left: 5px;"></i>
                    إجمالي الحسابات: <?php echo number_format($counter); ?> حساب
                </div>
                <div style="font-size: 13px; color: #7f8c8d;">
                    <i class="fas fa-calculator" style="margin-left: 5px;"></i>
                    إجمالي الأرصدة: <?php echo number_format($total_balance, 2); ?> ¥
                </div>
            </div>
            <?php else: ?>
                <div style="text-align: center; color: #7f8c8d; font-size: 13px;">
                    <i class="fas fa-info-circle" style="margin-left: 5px;"></i>
                    لا توجد حسابات بنكية لعرضها
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- نافذة إضافة/تعديل حساب -->
<div id="accountFormModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 600px; max-height: 90vh; overflow-y: auto; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-university" style="margin-left: 8px;"></i>
                    <span id="formModalTitle">إضافة حساب بنكي جديد</span>
                </h3>
                <button onclick="hideAccountFormModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px;">
            <form method="post" id="bankForm" onsubmit="return validateBankForm()">
                <input type="hidden" name="action" id="formAction" value="create_bank_account" />
                <input type="hidden" name="id" id="formAccountId" value="" />
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                            <span style="color: #e74c3c;">*</span> اسم الحساب
                        </label>
                        <input type="text" name="name" id="formName" class="form-control" 
                               style="width: 100%; padding: 8px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;"
                               required placeholder="مثال: الحساب الرئيسي" />
                    </div>
                    
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                            <span style="color: #e74c3c;">*</span> اسم البنك
                        </label>
                        <input type="text" name="bank_name" id="formBankName" class="form-control"
                               style="width: 100%; padding: 8px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;"
                               required placeholder="مثال: البنك الأهلي" />
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                            <span style="color: #e74c3c;">*</span> رقم الحساب
                        </label>
                        <input type="text" name="account_number" id="formAccountNumber" class="form-control"
                               style="width: 100%; padding: 8px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;"
                               required placeholder="مثال: 123456789" />
                        <small style="font-size: 11px; color: #7f8c8d;">سيتم التحقق من عدم تكرار رقم الحساب</small>
                    </div>
                    
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                            <span style="color: #e74c3c;">*</span> الفرع
                        </label>
                        <input type="text" name="branch_name" id="formBranchName" class="form-control"
                               style="width: 100%; padding: 8px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;"
                               required placeholder="مثال: فرع المدينة المنورة" />
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                            IBAN
                        </label>
                        <input type="text" name="iban" id="formIban" class="form-control"
                               style="width: 100%; padding: 8px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;"
                               placeholder="SAXXXXXXXXXXXXXXXXXXXXXXXX" />
                    </div>
                    
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                            SWIFT Code
                        </label>
                        <input type="text" name="swift_code" id="formSwiftCode" class="form-control"
                               style="width: 100%; padding: 8px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px;"
                               placeholder="مثال: NCBKSAJE" />
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50; font-size: 13px;">
                            <span style="color: #e74c3c;">*</span> الرصيد الافتتاحي
                        </label>
                        <div style="position: relative;">
                            <input type="number" name="opening_balance" id="formOpeningBalance" class="form-control"
                                   style="width: 100%; padding: 8px 12px; border: 1px solid #bdc3c7; border-radius: 0; font-size: 14px; padding-left: 40px;"
                                   step="0.01" required placeholder="0.00" value="0" />
                            <span style="position: absolute; left: 12px; top: 50%; transform: translateY(-50%); color: #7f8c8d;">¥</span>
                        </div>
                    </div>
                    
                    <div style="display: flex; align-items: center;">
                        <label style="display: flex; align-items: center; gap: 8px; font-weight: 600; color: #2c3e50; font-size: 13px; margin-top: 20px;">
                            <input type="checkbox" name="is_active" id="formIsActive" value="1" checked 
                                   style="width: 18px; height: 18px;" />
                            الحساب مفعل
                        </label>
                    </div>
                </div>
                
                <div style="display: flex; gap: 10px; justify-content: flex-end; margin-top: 25px; padding-top: 20px; border-top: 1px solid #ecf0f1;">
                    <button type="button" onclick="hideAccountFormModal()" 
                            style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                        إلغاء
                    </button>
                    <button type="submit" 
                            style="padding: 8px 20px; background: #2ecc71; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; display: inline-flex; align-items: center;">
                        <i class="fas fa-save" style="margin-left: 8px;"></i>
                        <span id="formSubmitText">حفظ الحساب</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- نافذة تأكيد الحذف -->
<div id="deleteModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 400px; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-trash" style="margin-left: 8px;"></i>
                    تأكيد الحذف
                </h3>
                <button onclick="hideDeleteModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px;">
            <div style="text-align: center; margin-bottom: 20px;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #e74c3c; margin-bottom: 15px;"></i>
                <p style="font-size: 16px; color: #2c3e50; margin-bottom: 10px;">هل أنت متأكد من حذف الحساب البنكي؟</p>
                <p style="font-size: 14px; color: #7f8c8d;" id="deleteMessage">
                    سيتم حذف الحساب البنكي <strong id="accountName"></strong> ولا يمكن التراجع عن هذا الإجراء.
                </p>
                <p style="font-size: 13px; color: #e74c3c; margin-top: 10px; padding: 8px; background: #fdedec; border-radius: 0;">
                    <i class="fas fa-exclamation-circle" style="margin-left: 5px;"></i>
                    لا يمكن حذف الحساب إذا كان مرتبطاً بمعاملات مالية.
                </p>
            </div>
            <div style="display: flex; gap: 10px; justify-content: flex-end;">
                <button onclick="hideDeleteModal()" style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                    إلغاء
                </button>
                <a href="#" class="btn" id="deleteLink"
                   style="padding: 8px 20px; background: #e74c3c; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center;">
                    نعم، احذف
                </a>
            </div>
        </div>
    </div>
</div>

<!-- نافذة عرض تفاصيل الحساب -->
<div id="viewAccountModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 500px; max-height: 90vh; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2); display: flex; flex-direction: column;">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-university" style="margin-left: 8px;"></i>
                    تفاصيل الحساب البنكي
                </h3>
                <button onclick="hideViewAccountModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px; overflow-y: auto; flex: 1;" id="accountDetails">
            <!-- سيتم تعبئة البيانات هنا عبر JavaScript -->
        </div>
        <div style="padding: 15px 20px; border-top: 1px solid #ecf0f1; background: #f8f9fa; display: flex; justify-content: flex-end; gap: 10px;">
            <button onclick="hideViewAccountModal()" style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                إغلاق
            </button>
        </div>
    </div>
</div>

<!-- نافذة تأكيد تغيير الحالة -->
<div id="statusModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000; align-items: center; justify-content: center;">
    <div style="background: white; width: 90%; max-width: 400px; border-radius: 0; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
        <div style="background: #2c3e50; color: white; padding: 15px 20px; border-bottom: 3px solid #1a252f;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <h3 style="margin: 0; font-weight: 600; font-size: 16px;">
                    <i class="fas fa-power-off" style="margin-left: 8px;"></i>
                    تغيير حالة الحساب
                </h3>
                <button onclick="hideStatusModal()" style="background: none; border: none; color: white; cursor: pointer; font-size: 18px;">×</button>
            </div>
        </div>
        <div style="padding: 20px;">
            <div style="text-align: center; margin-bottom: 20px;">
                <i class="fas fa-question-circle" style="font-size: 48px; color: #3498db; margin-bottom: 15px;"></i>
                <p style="font-size: 16px; color: #2c3e50; margin-bottom: 10px;" id="statusModalTitle">تأكيد تفعيل الحساب</p>
                <p style="font-size: 14px; color: #7f8c8d;" id="statusModalMessage">
                    هل أنت متأكد من تغيير حالة الحساب <strong id="statusAccountName"></strong>؟
                </p>
            </div>
            <div style="display: flex; gap: 10px; justify-content: flex-end;">
                <button onclick="hideStatusModal()" style="padding: 8px 20px; background: #95a5a6; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;">
                    إلغاء
                </button>
                <form method="POST" id="statusForm" style="margin: 0;">
                    <input type="hidden" name="action" value="update_bank_account_status">
                    <input type="hidden" name="account_id" id="statusAccountId">
                    <input type="hidden" name="new_status" id="newStatus">
                    <button type="submit" style="padding: 8px 20px; background: #3498db; color: white; border: none; border-radius: 0; font-size: 14px; cursor: pointer;" id="statusConfirmBtn">
                        نعم، تأكيد
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<style>
    /* تنسيقات إضافية متناسقة مع واجهة الفواتير */
    .form-control:focus {
        outline: none;
        border-color: #3498db;
        box-shadow: 0 0 0 2px rgba(52,152,219,0.2);
    }
    
    .table tbody tr {
        transition: background-color 0.2s ease;
    }
    
    .table tbody tr.selected {
        background-color: rgba(14, 66, 98, 0.1) !important;
        border-left: 3px solid #F19E2C;
    }
    
    /* تخصيص شريط التمرير */
    ::-webkit-scrollbar {
        width: 8px;
        height: 8px;
    }
    
    ::-webkit-scrollbar-track {
        background: #f1f1f1;
    }
    
    ::-webkit-scrollbar-thumb {
        background: #bdc3c7;
    }
    
    ::-webkit-scrollbar-thumb:hover {
        background: #95a5a6;
    }
    
    /* إخفاء بعض الأعمدة في الجوال */
    @media (max-width: 768px) {
        .table thead th:nth-child(5), /* الرصيد الحالي */
        .table tbody td:nth-child(5),
        .table thead th:nth-child(6), /* IBAN */
        .table tbody td:nth-child(6) {
            display: none;
        }
        
        .table thead th,
        .table tbody td {
            padding: 8px 5px !important;
        }
        
        #accountFormModal > div {
            width: 95% !important;
        }
        
        #accountFormModal > div > div:last-child {
            padding: 15px !important;
        }
        
        #accountFormModal form > div {
            grid-template-columns: 1fr !important;
            gap: 10px !important;
        }
    }
</style>

<script>
// متغيرات عالمية لتخزين الحساب المحدد
let selectedAccountId = null;
let selectedAccountName = null;
let selectedAccountElement = null;
let selectedAccountStatus = null;

// اختيار حساب
function selectAccount(accountId, element) {
    // إلغاء تحديد الصف السابق إذا كان موجوداً
    if (selectedAccountElement) {
        selectedAccountElement.classList.remove('selected');
    }

    // تحديد الصف الجديد
    element.classList.add('selected');
    selectedAccountElement = element;

    // حفظ بيانات الحساب المحدد
    selectedAccountId = accountId;
    selectedAccountName = element.getAttribute('data-account-name');
    selectedAccountStatus = element.getAttribute('data-account-status');

    // تحديث روابط الأزرار
    updateActionButtons();

    // إظهار حاوية الأزرار
    showActionsContainer();

    // تمرير سلس إلى الصف المحدد في الجوال
    if (window.innerWidth <= 768) {
        setTimeout(() => {
            const actionsContainer = document.getElementById('actionsContainer');
            actionsContainer.style.maxHeight = actionsContainer.scrollHeight + 'px';
            actionsContainer.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }, 300);
    }
}

// تحديث روابط الأزرار بناءً على الحساب المحدد
function updateActionButtons() {
    if (!selectedAccountId) return;

    // تحديث روابط الأزرار
    document.getElementById('editBtn').onclick = () => editAccount(selectedAccountId);
    document.getElementById('viewBtn').onclick = () => viewAccount(selectedAccountId);

    document.getElementById('toggleStatusBtn').onclick = () => toggleAccountStatus(selectedAccountId, selectedAccountStatus);
    document.getElementById('deleteBtn').onclick = () => confirmDelete(selectedAccountId, selectedAccountName);

    // تحديث زر تغيير الحالة
    updateStatusButton();
}

// تحديث زر تغيير الحالة
function updateStatusButton() {
    const statusBtn = document.getElementById('toggleStatusBtn');
    const statusBtnText = document.getElementById('toggleStatusText');
    
    if (!selectedAccountStatus) return;

    if (selectedAccountStatus === '0') {
        statusBtn.style.backgroundColor = '#28a745';
        statusBtn.onmouseover = function() { this.style.backgroundColor = '#218838'; };
        statusBtn.onmouseout = function() { this.style.backgroundColor = '#28a745'; };
        statusBtnText.textContent = 'تفعيل';
    } else {
        statusBtn.style.backgroundColor = '#f39c12';
        statusBtn.onmouseover = function() { this.style.backgroundColor = '#e67e22'; };
        statusBtn.onmouseout = function() { this.style.backgroundColor = '#f39c12'; };
        statusBtnText.textContent = 'تعطيل';
    }
}

// إظهار حاوية أزرار الإجراءات
function showActionsContainer() {
    const container = document.getElementById('actionsContainer');
    container.style.maxHeight = container.scrollHeight + 'px';
}

// إخفاء حاوية أزرار الإجراءات
function hideActionsContainer() {
    const container = document.getElementById('actionsContainer');
    container.style.maxHeight = '0';
}

// إظهار نموذج إضافة/تعديل الحساب
function showAddForm() {
    document.getElementById('formModalTitle').textContent = 'إضافة حساب بنكي جديد';
    document.getElementById('formAction').value = 'create_bank_account';
    document.getElementById('formSubmitText').textContent = 'حفظ الحساب';
    document.getElementById('formAccountId').value = '';
    document.getElementById('formName').value = '';
    document.getElementById('formBankName').value = '';
    document.getElementById('formAccountNumber').value = '';
    document.getElementById('formBranchName').value = '';
    document.getElementById('formIban').value = '';
    document.getElementById('formSwiftCode').value = '';
    document.getElementById('formOpeningBalance').value = '0';
    document.getElementById('formIsActive').checked = true;
    
    document.getElementById('accountFormModal').style.display = 'flex';
}

// تعديل حساب
function editAccount(accountId) {
    const row = document.querySelector(`tr[data-account-id="${accountId}"]`);
    if (!row) return;

    document.getElementById('formModalTitle').textContent = 'تعديل حساب بنكي';
    document.getElementById('formAction').value = 'update_bank_account';
    document.getElementById('formSubmitText').textContent = 'تحديث الحساب';
    document.getElementById('formAccountId').value = accountId;
    document.getElementById('formName').value = row.getAttribute('data-account-name');
    document.getElementById('formBankName').value = row.getAttribute('data-bank-name');
    document.getElementById('formAccountNumber').value = row.getAttribute('data-account-number');
    document.getElementById('formBranchName').value = row.getAttribute('data-branch');
    document.getElementById('formIban').value = row.getAttribute('data-iban');
    document.getElementById('formSwiftCode').value = row.getAttribute('data-swift');
    document.getElementById('formOpeningBalance').value = row.getAttribute('data-account-balance');
    document.getElementById('formIsActive').checked = row.getAttribute('data-account-status') === '1';
    
    document.getElementById('accountFormModal').style.display = 'flex';
}

// عرض تفاصيل الحساب
function viewAccount(accountId) {
    const row = document.querySelector(`tr[data-account-id="${accountId}"]`);
    if (!row) return;

    const status = row.getAttribute('data-account-status') === '1' ? 
        '<span style="background: #2ecc71; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">مفعل</span>' : 
        '<span style="background: #e74c3c; color: white; padding: 4px 10px; border-radius: 2px; font-size: 12px; font-weight: 600;">غير مفعل</span>';

    const iban = row.getAttribute('data-iban') ? 
        `<code style="background: #e8f6f3; padding: 4px 8px; border-radius: 2px; font-size: 12px; color: #16a085;">${row.getAttribute('data-iban')}</code>` : 
        '<span style="color: #95a5a6;">غير محدد</span>';

    const swift = row.getAttribute('data-swift') ? 
        `<code style="background: #e8f4fd; padding: 4px 8px; border-radius: 2px; font-size: 12px; color: #3498db;">${row.getAttribute('data-swift')}</code>` : 
        '<span style="color: #95a5a6;">غير محدد</span>';

    document.getElementById('accountDetails').innerHTML = `
        <div style="margin-bottom: 20px;">
            <h4 style="color: #2c3e50; border-bottom: 2px solid #F19E2C; padding-bottom: 8px; margin-bottom: 15px;">
                <i class="fas fa-university" style="margin-left: 8px;"></i>
                معلومات الحساب البنكي
            </h4>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #3498db;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">اسم الحساب</div>
                    <div style="font-size: 16px; font-weight: 600; color: #2c3e50;">${row.getAttribute('data-account-name')}</div>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #2ecc71;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">البنك</div>
                    <div style="font-size: 16px; font-weight: 600; color: #2c3e50;">${row.getAttribute('data-bank-name')}</div>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #9b59b6;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">رقم الحساب</div>
                    <div style="font-size: 16px; font-weight: 600; color: #2c3e50;">
                        <code style="background: #e8f4fd; padding: 4px 8px; border-radius: 2px; font-size: 14px;">${row.getAttribute('data-account-number')}</code>
                    </div>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #f39c12;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">الحالة</div>
                    <div style="font-size: 16px; font-weight: 600;">${status}</div>
                </div>
            </div>
        </div>
        
        <div style="margin-bottom: 20px;">
            <h4 style="color: #2c3e50; border-bottom: 2px solid #F19E2C; padding-bottom: 8px; margin-bottom: 15px;">
                <i class="fas fa-money-bill-wave" style="margin-left: 8px;"></i>
                المعلومات المالية
            </h4>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div style="background: #e8f6f3; padding: 15px; border-left: 3px solid #1abc9c;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">الرصيد الحالي</div>
                    <div style="font-size: 20px; font-weight: 600; color: #27ae60;">
                        ${parseFloat(row.getAttribute('data-account-balance')).toFixed(2)} ¥
                    </div>
                </div>
                
                <div style="background: #fef9e7; padding: 15px; border-left: 3px solid #f1c40f;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">الفرع</div>
                    <div style="font-size: 16px; font-weight: 600; color: #f39c12;">
                        ${row.getAttribute('data-branch')}
                    </div>
                </div>
            </div>
        </div>
        
        <div>
            <h4 style="color: #2c3e50; border-bottom: 2px solid #F19E2C; padding-bottom: 8px; margin-bottom: 15px;">
                <i class="fas fa-info-circle" style="margin-left: 8px;"></i>
                معلومات إضافية
            </h4>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #3498db;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">IBAN</div>
                    <div style="font-size: 14px; font-weight: 500; color: #2c3e50;">
                        ${iban}
                    </div>
                </div>
                
                <div style="background: #f8f9fa; padding: 15px; border-left: 3px solid #9b59b6;">
                    <div style="font-size: 12px; color: #7f8c8d; margin-bottom: 5px;">SWIFT Code</div>
                    <div style="font-size: 14px; font-weight: 500; color: #2c3e50;">
                        ${swift}
                    </div>
                </div>
            </div>
        </div>
    `;

    document.getElementById('viewAccountModal').style.display = 'flex';
}

// تغيير حالة الحساب (تفعيل/تعطيل)
function toggleAccountStatus(accountId, currentStatus) {
    const row = document.querySelector(`tr[data-account-id="${accountId}"]`);
    if (!row) return;

    const newStatus = currentStatus === '1' ? '0' : '1';
    const actionText = newStatus === '1' ? 'تفعيل' : 'تعطيل';
    
    document.getElementById('statusAccountId').value = accountId;
    document.getElementById('newStatus').value = newStatus;
    document.getElementById('statusAccountName').textContent = row.getAttribute('data-account-name');
    document.getElementById('statusModalTitle').textContent = 'تأكيد ' + actionText + ' الحساب';
    document.getElementById('statusModalMessage').innerHTML =
        `هل أنت متأكد من ${actionText} الحساب <strong>${row.getAttribute('data-account-name')}</strong>؟`;
    document.getElementById('statusConfirmBtn').textContent = 'نعم، ' + actionText;

    document.getElementById('statusModal').style.display = 'flex';
}

// تأكيد حذف الحساب
function confirmDelete(accountId, accountName) {
    document.getElementById('accountName').textContent = accountName;
    document.getElementById('deleteMessage').innerHTML =
        `سيتم حذف الحساب البنكي <strong>${accountName}</strong> ولا يمكن التراجع عن هذا الإجراء.`;
    
    document.getElementById('deleteLink').href = `bank_accounts.php?delete=${accountId}`;
    
    document.getElementById('deleteModal').style.display = 'flex';
}

// التحقق من صحة النموذج
function validateBankForm() {
    const accountNumber = document.getElementById('formAccountNumber').value.trim();
    const openingBalance = parseFloat(document.getElementById('formOpeningBalance').value) || 0;
    
    // التحقق من رقم الحساب
    if (accountNumber.length < 3) {
        alert('رقم الحساب يجب أن يكون على الأقل 3 أحرف');
        return false;
    }
    
    // التحقق من الرصيد الافتتاحي
    if (openingBalance < 0) {
        alert('الرصيد الافتتاحي لا يمكن أن يكون سالباً');
        return false;
    }
    
    return true;
}

// وظائف إخفاء النوافذ المنبثقة
function hideAccountFormModal() {
    document.getElementById('accountFormModal').style.display = 'none';
}

function hideDeleteModal() {
    document.getElementById('deleteModal').style.display = 'none';
}

function hideViewAccountModal() {
    document.getElementById('viewAccountModal').style.display = 'none';
}

function hideStatusModal() {
    document.getElementById('statusModal').style.display = 'none';
}

// إغلاق النوافذ المنبثقة عند النقر خارجها
window.onclick = function(event) {
    const modals = ['accountFormModal', 'deleteModal', 'viewAccountModal', 'statusModal'];
    modals.forEach(modalId => {
        const modal = document.getElementById(modalId);
        if (event.target == modal) {
            modal.style.display = 'none';
        }
    });
};

// إضافة تأثيرات تفاعلية عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    // تنسيق حقل IBAN أثناء الكتابة
    const ibanInput = document.getElementById('formIban');
    if (ibanInput) {
        ibanInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/[^A-Z0-9]/gi, '').toUpperCase();
            e.target.value = value;
        });
    }

    // إخفاء حاوية الأزرار عند النقر خارج الجدول
    document.addEventListener('click', function(event) {
        const table = document.querySelector('.table');
        const actionsContainer = document.getElementById('actionsContainer');

        // التحقق إذا كان النقر على أي من أزرار الإجراءات أو داخل النافذة المنبثقة
        const isModal = event.target.closest('#accountFormModal') || 
                        event.target.closest('#deleteModal') ||
                        event.target.closest('#viewAccountModal') ||
                        event.target.closest('#statusModal');
        
        const isActionButton = event.target.closest('#actionsContainer') || 
                              event.target.closest('.btn') ||
                              isModal;

        // إذا تم النقر خارج الجدول وخارج حاوية الأزرار وخارج النوافذ المنبثقة
        if (!table.contains(event.target) && !isActionButton) {
            if (selectedAccountElement) {
                selectedAccountElement.classList.remove('selected');
                selectedAccountElement = null;
                selectedAccountId = null;
                selectedAccountName = null;
                selectedAccountStatus = null;
                hideActionsContainer();
            }
        }
    });

    // إضافة وظيفة البحث السريع
    const searchInput = document.querySelector('input[name="search"]');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('.table tbody tr');
            
            rows.forEach(row => {
                const accountName = row.getAttribute('data-account-name') || '';
                const bankName = row.getAttribute('data-bank-name') || '';
                const accountNumber = row.getAttribute('data-account-number') || '';
                
                if (accountName.toLowerCase().includes(searchTerm) || 
                    bankName.toLowerCase().includes(searchTerm) ||
                    accountNumber.toLowerCase().includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
    }
});
</script>

