<?php
/*
  ملف واحد لإدارة الصناديق والبنوك وسندات القبض والصرف
  - يستخدم HTML/CSS/JS/PHP (بدون روتينج) -> كل العمليات تنفذ عبر POST إلى نفس الملف
  - متوافق مع هيكل قاعدة البيانات المرسلة (الجداول: cash_boxes, bank_accounts, vouchers, transactions, customers, companies, activity_logs)
  - عند إنشاء / تعديل / إيقاف الصناديق والبنوك: يتم تسجيل رصيد افتتاحي (كتحويل إلى transactions) وتحديث current_balance
  - عند إنشاء سند قبض/صرف: يتم إنشاء قيد محاسبي في transactions لكل طرف (الصندوق/البنك + الطرف المقابل customer/company)
  ملاحظات: احتفظ بنسخة احتياطية من قاعدة البيانات قبل التشغيل.
*/

// -------- إعداد الاتصال بقاعدة البيانات --------
$DB_HOST = '127.0.0.1';
$DB_USER = 'goldhldz_aknan';
$DB_PASS = '';// عدِّله حسب بيئتك
$DB_NAME = 'goldhldz_onlinedesign';

$mysqli = new mysqli($DB_HOST, $DB_USER, $DB_PASS, $DB_NAME);
if ($mysqli->connect_errno) {
    die("فشل الاتصال بقاعدة البيانات: " . $mysqli->connect_error);
}
$mysqli->set_charset('utf8mb4');

// ---------- دوال مساعدة عامة ----------
function json_post($k){ return isset($_POST[$k]) ? $_POST[$k] : null; }
function now(){ return date('Y-m-d H:i:s'); }

// استرجاع آخر رصيد لحساب معين (account_type, account_id) لحساب balance_after
function get_last_balance($mysqli, $account_type, $account_id){
    $sql = "SELECT balance_after FROM transactions WHERE account_type=? AND account_id=? ORDER BY id DESC LIMIT 1";
    $stmt = $mysqli->prepare($sql);
    $stmt->bind_param('si', $account_type, $account_id);
    $stmt->execute();
    $stmt->bind_result($balance);
    if($stmt->fetch()){ $stmt->close(); return (float)$balance; }
    $stmt->close();
    return 0.00;
}

// إدراج عملية في جدول transactions
function insert_transaction($mysqli, $date, $account_type, $voucher_type, $account_id, $related_invoice_id, $related_voucher_id, $debit, $credit, $description, $payment_method){
    // حساب balance_after بناءً على آخر رصيد
    $last = get_last_balance($mysqli, $account_type, $account_id);
    // المنطق: for cash/bank accounts, debit increases balance, credit decreases
    $new_balance = $last + (float)$debit - (float)$credit;
    $sql = "INSERT INTO transactions (date, account_type, voucher_type, account_id, related_invoice_id, related_voucher_id, debit_amount, credit_amount, description, balance_after, created_at, payment_method)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $mysqli->prepare($sql);
    $created_at = date('Y-m-d H:i:s');
    $stmt->bind_param('sssiiiddsdss', $date, $account_type, $voucher_type, $account_id, $related_invoice_id, $related_voucher_id, $debit, $credit, $description, $new_balance, $created_at, $payment_method);
    if(!$stmt->execute()){
        error_log('insert_transaction error: '.$stmt->error);
    }
    $stmt->close();
    return $new_balance;
}

// تسجيل نشاط في activity_logs
function log_activity($mysqli, $user_id, $action, $entity_type, $entity_id, $old_values, $new_values){
    $sql = "INSERT INTO activity_logs (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $mysqli->prepare($sql);
    $ip = $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $created_at = date('Y-m-d H:i:s');
    $stmt->bind_param('ississsss', $user_id, $action, $entity_type, $entity_id, $old_values, $new_values, $ip, $ua, $created_at);
    $stmt->execute();
    $stmt->close();
}

// تحديث current_balance في cash_boxes أو bank_accounts
function update_container_balance($mysqli, $table, $id, $new_balance){
    $sql = "UPDATE $table SET current_balance = ?, updated_at = ? WHERE id = ?";
    $stmt = $mysqli->prepare($sql);
    $t = date('Y-m-d H:i:s');
    $stmt->bind_param('dsi', $new_balance, $t, $id);
    $stmt->execute();
    $stmt->close();
}

// دالة مساعدة للحصول على اسم مستخدم تجريبي (في بيئة حقيقية استبدل بجلسة المستخدم الحقيقية)
function current_user_id(){
    return 1; // افتراضياً admin
}

// ---------- معالجة العمليات الواردة عبر POST ----------
$action = $_POST['action'] ?? '';
$message = '';

if($action === 'create_cash_box'){
    $name = trim($_POST['name']);
    $description = $_POST['description'] ?? '';
    $opening = (float)$_POST['opening_balance'];
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    // إنشاء الصندوق
    $sql = "INSERT INTO cash_boxes (name, description, opening_balance, current_balance, is_active, created_at) VALUES (?, ?, ?, ?, ?, ?)";
    $stmt = $mysqli->prepare($sql);
    $t = date('Y-m-d H:i:s');
    $stmt->bind_param('ssddis', $name, $description, $opening, $opening, $is_active, $t);
    $stmt->execute();
    $cash_id = $stmt->insert_id;
    $stmt->close();
    // تسجيل قيد افتتاحي في transactions (receipt)
    if($opening != 0){
        $desc = "رصيد افتتاحي للصندوق: $name";
        $newbal = insert_transaction($mysqli, date('Y-m-d'), 'cash', 'receipt', $cash_id, null, null, $opening, 0.00, $desc, 'cash');
        update_container_balance($mysqli, 'cash_boxes', $cash_id, $newbal);
    }
    // activity log
    log_activity($mysqli, current_user_id(), 'create', 'cash_box', $cash_id, null, json_encode(['name'=>$name,'opening_balance'=>$opening]));
    $message = 'تم إنشاء الصندوق بنجاح.';
}

if($action === 'update_cash_box'){
    $id = (int)$_POST['id'];
    $name = trim($_POST['name']);
    $description = $_POST['description'] ?? '';
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    // احصل على القيم القديمة
    $res = $mysqli->query("SELECT name, description, opening_balance, current_balance, is_active FROM cash_boxes WHERE id=$id LIMIT 1");
    $old = $res->fetch_assoc();
    $sql = "UPDATE cash_boxes SET name=?, description=?, is_active=?, updated_at=? WHERE id=?";
    $stmt = $mysqli->prepare($sql);
    $t = date('Y-m-d H:i:s');
    $stmt->bind_param('sisis', $name, $description, $is_active, $t, $id);
    $stmt->execute();
    $stmt->close();
    log_activity($mysqli, current_user_id(), 'update', 'cash_box', $id, json_encode($old), json_encode(['name'=>$name,'is_active'=>$is_active]));
    $message = 'تم تحديث بيانات الصندوق.';
}

if($action === 'create_bank_account'){
    $name = trim($_POST['name']);
    $bank_name = trim($_POST['bank_name']);
    $account_number = trim($_POST['account_number']);
    $branch = trim($_POST['branch_name']);
    $iban = trim($_POST['iban']);
    $swift = trim($_POST['swift_code']);
    $opening = (float)$_POST['opening_balance'];
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    $sql = "INSERT INTO bank_accounts (name, bank_name, account_number, branch_name, iban, swift_code, opening_balance, current_balance, is_active, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $mysqli->prepare($sql);
    $t = date('Y-m-d H:i:s');
    $stmt->bind_param('ssssssddis', $name, $bank_name, $account_number, $branch, $iban, $swift, $opening, $opening, $is_active, $t);
    $stmt->execute();
    $bank_id = $stmt->insert_id;
    $stmt->close();
    if($opening != 0){
        $desc = "رصيد افتتاحي للحساب البنكي: $name";
        $newbal = insert_transaction($mysqli, date('Y-m-d'), 'bank', 'receipt', $bank_id, null, null, $opening, 0.00, $desc, 'bank');
        update_container_balance($mysqli, 'bank_accounts', $bank_id, $newbal);
    }
    log_activity($mysqli, current_user_id(), 'create', 'bank_account', $bank_id, null, json_encode(['name'=>$name,'opening_balance'=>$opening]));
    $message = 'تم إنشاء الحساب البنكي.';
}

if($action === 'update_bank_account'){
    $id = (int)$_POST['id'];
    $name = trim($_POST['name']);
    $bank_name = trim($_POST['bank_name']);
    $account_number = trim($_POST['account_number']);
    $branch = trim($_POST['branch_name']);
    $iban = trim($_POST['iban']);
    $swift = trim($_POST['swift_code']);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    $res = $mysqli->query("SELECT name, bank_name, account_number, branch_name, opening_balance, current_balance, is_active FROM bank_accounts WHERE id=$id LIMIT 1");
    $old = $res->fetch_assoc();
    $sql = "UPDATE bank_accounts SET name=?, bank_name=?, account_number=?, branch_name=?, iban=?, swift_code=?, is_active=?, updated_at=? WHERE id=?";
    $stmt = $mysqli->prepare($sql);
    $t = date('Y-m-d H:i:s');
    $stmt->bind_param('sssssisis', $name, $bank_name, $account_number, $branch, $iban, $swift, $is_active, $t, $id);
    $stmt->execute();
    $stmt->close();
    log_activity($mysqli, current_user_id(), 'update', 'bank_account', $id, json_encode($old), json_encode(['name'=>$name,'is_active'=>$is_active]));
    $message = 'تم تحديث بيانات الحساب البنكي.';
}

// إنشاء سند قبض/صرف
if($action === 'create_voucher'){
    /* متغيرات متوقعة في POST:
       type: receipt|payment
       amount
       currency_id
       exchange_rate
       related_to: customer|company
       related_id
       payment_type: cash|bank|both
       cash_box_id (if cash or both)
       bank_account_id (if bank or both)
       description
    */
    $type = $_POST['type'];
    $amount = (float)$_POST['amount'];
    $currency_id = (int)$_POST['currency_id'];
    $exchange_rate = (float)$_POST['exchange_rate'];
    $related_to = $_POST['related_to'];
    $related_id = (int)$_POST['related_id'];
    $payment_type = $_POST['payment_type'];
    $cash_box_id = isset($_POST['cash_box_id']) ? (int)$_POST['cash_box_id'] : null;
    $bank_account_id = isset($_POST['bank_account_id']) ? (int)$_POST['bank_account_id'] : null;
    $description = $_POST['description'] ?? '';
    $voucher_number = 'VCHR-'.date('YmdHis').rand(100,999);
    $created_by = current_user_id();
    // إدراج السند
    $sql = "INSERT INTO vouchers (voucher_number, type, amount, currency_id, exchange_rate, related_to, related_id, cash_box_id, bank_account_id, description, created_by, created_at, payment_type, voucher_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $mysqli->prepare($sql);
    $t = date('Y-m-d H:i:s');
    $voucher_date = $t;
    $stmt->bind_param('ssddisiiisssis', $voucher_number, $type, $amount, $currency_id, $exchange_rate, $related_to, $related_id, $cash_box_id, $bank_account_id, $description, $created_by, $t, $payment_type, $voucher_date);
    if(!$stmt->execute()){
        $message = 'خطأ عند حفظ السند: '.$stmt->error;
    } else {
        $voucher_id = $stmt->insert_id;
        $stmt->close();
        // الآن تسجيل القيود في transactions
        // الحالة: سند قبض (receipt) => نقد/بنك يستلم (debit on cash/bank), والطرف المقابل (customer/company) يُقيد بالعكس (credit)
        // سند صرف (payment) => نقد/بنك يتم سحب (credit on cash/bank), والطرف المقابل يدخل مدين (debit)

        // دعم payment_type both: نقسم المبلغ بحسب المدخلات (cash_amount, bank_amount) إن أرسلت
        $cash_amount = 0.00; $bank_amount = 0.00;
        if($payment_type === 'cash') $cash_amount = $amount;
        elseif($payment_type === 'bank') $bank_amount = $amount;
        elseif($payment_type === 'both'){
            $cash_amount = (float)($_POST['cash_amount'] ?? 0);
            $bank_amount = $amount - $cash_amount;
        }

        // helper to apply transaction to container and update its current_balance
        $apply_container = function($container_type, $container_id, $debit, $credit, $desc, $payment_method) use ($mysqli, $voucher_id){
            // insert and update container balance
            $newbal = insert_transaction($mysqli, date('Y-m-d'), $container_type, 'receipt', $container_id, null, $voucher_id, $debit, $credit, $desc, $payment_method);
            if($container_type === 'cash') update_container_balance($mysqli, 'cash_boxes', $container_id, $newbal);
            if($container_type === 'bank') update_container_balance($mysqli, 'bank_accounts', $container_id, $newbal);
        };

        // وصف الطرف المقابل
        $related_name = '';
        if($related_to === 'customer'){
            $r = $mysqli->query("SELECT name FROM customers WHERE id=$related_id LIMIT 1");
            $related_name = $r->fetch_row()[0] ?? '';
        } else {
            $r = $mysqli->query("SELECT name FROM companies WHERE id=$related_id LIMIT 1");
            $related_name = $r->fetch_row()[0] ?? '';
        }

        if($type === 'receipt'){
            // نقد/بنك يستلم => debit on cash/bank, and credit on related account
            if($cash_amount > 0 && $cash_box_id){
                $desc = "سند قبض من {$related_to} ($related_name) - $voucher_number";
                $apply_container('cash', $cash_box_id, $cash_amount, 0.00, $desc, 'cash');
            }
            if($bank_amount > 0 && $bank_account_id){
                $desc = "سند قبض من {$related_to} ($related_name) - $voucher_number";
                $apply_container('bank', $bank_account_id, $bank_amount, 0.00, $desc, 'bank');
            }
            // للطرف المقابل (customer/company) نضع قيد معاكس: credit يقلل من مدينته
            $related_desc = "سند قبض #$voucher_number - {$description}";
            // customer/company account_type uses same insert_transaction but account_type param accordingly
            // for related: credit for customer/company when receipt
            insert_transaction($mysqli, date('Y-m-d'), $related_to, null, $related_id, null, $voucher_id, 0.00, $amount, $related_desc, 'cash');
            // تحديث current_balance في customers/companies
            // نحسب آخر رصيد لهم
            $last = get_last_balance($mysqli, $related_to, $related_id);
            // تحديث الحقل current_balance في الجدول المناسب
         if($related_to === 'customer'){

    $new_balance = $last - $amount;   // ✅ نحول العملية إلى متغير

    $upd = $mysqli->prepare("UPDATE customers SET current_balance=? WHERE id=?");
    $upd->bind_param('di', $new_balance, $related_id);
    $upd->execute();
    $upd->close();

} else {

    $new_balance = $last - $amount;   // ✅ نفس الشيء هنا

    $upd = $mysqli->prepare("UPDATE companies SET current_balance=? WHERE id=?");
    $upd->bind_param('di', $new_balance, $related_id);
    $upd->execute();
    $upd->close();

}

        } else { // payment
            // نقد/بنك يُسحب => credit on cash/bank and debit on related
            if($cash_amount > 0 && $cash_box_id){
                $desc = "سند صرف الى {$related_to} ($related_name) - $voucher_number";
                $apply_container('cash', $cash_box_id, 0.00, $cash_amount, $desc, 'cash');
            }
            if($bank_amount > 0 && $bank_account_id){
                $desc = "سند صرف الى {$related_to} ($related_name) - $voucher_number";
                $apply_container('bank', $bank_account_id, 0.00, $bank_amount, $desc, 'bank');
            }
            $related_desc = "سند صرف #$voucher_number - {$description}";
            insert_transaction($mysqli, date('Y-m-d'), $related_to, null, $related_id, null, $voucher_id, $amount, 0.00, $related_desc, 'cash');
            $last = get_last_balance($mysqli, $related_to, $related_id);
            if($related_to === 'customer'){
                $upd = $mysqli->prepare("UPDATE customers SET current_balance=? WHERE id=?");
                $upd->bind_param('di', $last + $amount, $related_id);
                $upd->execute(); $upd->close();
            } else {
                $upd = $mysqli->prepare("UPDATE companies SET current_balance=? WHERE id=?");
                $upd->bind_param('di', $last + $amount, $related_id);
                $upd->execute(); $upd->close();
            }
        }

        log_activity($mysqli, current_user_id(), 'create', 'voucher', $voucher_id, null, json_encode(['voucher_number'=>$voucher_number,'amount'=>$amount,'type'=>$type]));
        $message = 'تم حفظ السند والمقيد المحاسبي بنجاح.';
    }
}

// ---------- استعلامات لعرض القوائم في الواجهة ----------
$cash_boxes = $mysqli->query("SELECT * FROM cash_boxes ORDER BY id DESC");
$bank_accounts = $mysqli->query("SELECT * FROM bank_accounts ORDER BY id DESC");
$customers = $mysqli->query("SELECT id, name FROM customers WHERE is_active=1 ORDER BY name");
$companies = $mysqli->query("SELECT id, name FROM companies WHERE is_active=1 ORDER BY name");
$currencies = $mysqli->query("SELECT id, name, code, symbol FROM currencies ORDER BY id");

// استعلام للسندات الحديثة
$recent_vouchers = $mysqli->query("SELECT v.*, 
                                   CASE WHEN v.related_to = 'customer' THEN c.name ELSE co.name END as related_name
                                   FROM vouchers v
                                   LEFT JOIN customers c ON v.related_to = 'customer' AND v.related_id = c.id
                                   LEFT JOIN companies co ON v.related_to = 'company' AND v.related_id = co.id
                                   ORDER BY v.id DESC LIMIT 10");
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>إدارة الصناديق والبنوك وسندات القبض/الصرف</title>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<style>
    /* تحسينات التصميم */
    :root {
    --primary-color: #0E4262;
            --secondary-color: #F19E2C;
        --success-color: #4cc9f0;
        --danger-color: #f72585;
        --warning-color: #f8961e;
        --light-color: #f8f9fa;
        --dark-color: #212529;
        --border-radius: 12px;
        --box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        --transition: all 0.3s ease;
    }

    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
    }

    body {
        font-family: 'Segoe UI', Tahoma, Arial, sans-serif;
        background: linear-gradient(135deg, #f5f7fa 0%, #e4edf5 100%);
        color: #333;
        line-height: 1.6;
        padding: 0;
        min-height: 100vh;
    }

    .container {
        max-width: 1300px;
        margin: 0 auto;
        padding: 20px;
    }

    .header {
        background: linear-gradient(to right, var(--primary-color), var(--secondary-color));
        color: white;
        padding: 20px 30px;
        border-radius: var(--border-radius);
        margin-bottom: 30px;
        box-shadow: var(--box-shadow);
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .header h1 {
        font-size: 1.8rem;
        display: flex;
        align-items: center;
        gap: 15px;
    }

    .header h1 i {
        font-size: 2rem;
    }

    .card {
        background: white;
        padding: 25px;
        border-radius: var(--border-radius);
        box-shadow: var(--box-shadow);
        margin-bottom: 25px;
        transition: var(--transition);
        border: 1px solid rgba(0,0,0,0.05);
    }

    .card:hover {
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.12);
    }

    .card-title {
        font-size: 1.4rem;
        color: var(--primary-color);
        margin-bottom: 20px;
        padding-bottom: 10px;
        border-bottom: 2px solid #f0f0f0;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .card-title i {
        color: var(--secondary-color);
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-label {
        display: block;
        margin-bottom: 8px;
        font-weight: 600;
        color: #555;
    }

    .form-control {
        width: 100%;
        padding: 12px 15px;
        border: 1px solid #ddd;
        border-radius: 8px;
        font-size: 1rem;
        transition: var(--transition);
    }

    .form-control:focus {
        outline: none;
        border-color: var(--primary-color);
        box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1);
    }

    .row {
        display: flex;
        flex-wrap: wrap;
        margin: 0 -10px;
    }

    .col {
        flex: 1;
        padding: 0 10px;
        min-width: 250px;
    }

    .btn {
        display: inline-block;
        padding: 12px 24px;
        background: var(--primary-color);
        color: white;
        border: none;
        border-radius: 8px;
        font-size: 1rem;
        font-weight: 600;
        cursor: pointer;
        transition: var(--transition);
        text-align: center;
    }

    .btn:hover {
        background: var(--secondary-color);
        transform: translateY(-2px);
        box-shadow: 0 6px 15px rgba(67, 97, 238, 0.25);
    }

    .btn-success {
        background: var(--success-color);
    }

    .btn-success:hover {
        background: #3aa8d8;
    }

    .btn-danger {
        background: var(--danger-color);
    }

    .btn-danger:hover {
        background: #e11570;
    }

    .btn-block {
        width: 100%;
        margin-top: 10px;
    }

    .alert {
        padding: 15px 20px;
        border-radius: 8px;
        margin-bottom: 25px;
        display: flex;
        align-items: center;
        gap: 12px;
    }

    .alert-success {
        background: #e8f7e8;
        border: 1px solid #c8e6c9;
        color: #2d572c;
    }

    .alert-error {
        background: #fdecea;
        border: 1px solid #f5c6cb;
        color: #721c24;
    }

    .tabs {
        display: flex;
        background: white;
        border-radius: var(--border-radius);
        overflow: hidden;
        margin-bottom: 25px;
        box-shadow: var(--box-shadow);
    }

    .tab {
        flex: 1;
        padding: 15px 20px;
        text-align: center;
        background: #f8f9fa;
        cursor: pointer;
        transition: var(--transition);
        border-right: 1px solid #eee;
        font-weight: 600;
        color: #666;
    }

    .tab:last-child {
        border-right: none;
    }

    .tab:hover {
        background: #e9ecef;
    }

    .tab.active {
        background: var(--primary-color);
        color: white;
    }

    .tab-content {
        display: none;
    }

    .tab-content.active {
        display: block;
    }

    .table-container {
        overflow-x: auto;
        border-radius: var(--border-radius);
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }

    table {
        width: 100%;
        border-collapse: collapse;
        background: white;
        min-width: 800px;
    }

    thead {
        background: linear-gradient(to right, var(--primary-color), var(--secondary-color));
        color: white;
    }

    th {
        padding: 15px;
        text-align: right;
        font-weight: 600;
    }

    tbody tr {
        border-bottom: 1px solid #f0f0f0;
        transition: var(--transition);
    }

    tbody tr:hover {
        background: #f8fafd;
    }

    td {
        padding: 15px;
        text-align: right;
    }

    .status-active {
        display: inline-block;
        padding: 5px 12px;
        background: #e8f7e8;
        color: #2d572c;
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 600;
    }

    .status-inactive {
        display: inline-block;
        padding: 5px 12px;
        background: #fdecea;
        color: #721c24;
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 600;
    }

    .amount-positive {
        color: #2d572c;
        font-weight: 700;
    }

    .amount-negative {
        color: #721c24;
        font-weight: 700;
    }

    .balance-display {
        font-size: 1.8rem;
        font-weight: 700;
        color: var(--primary-color);
        text-align: center;
        padding: 20px;
        background: linear-gradient(135deg, #f8fafd 0%, #eef2f7 100%);
        border-radius: var(--border-radius);
        margin: 15px 0;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }

    .stat-card {
        background: white;
        padding: 20px;
        border-radius: var(--border-radius);
        box-shadow: var(--box-shadow);
        display: flex;
        align-items: center;
        gap: 20px;
    }

    .stat-icon {
        width: 60px;
        height: 60px;
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.8rem;
        color: white;
    }

    .cash-icon {
        background: linear-gradient(135deg, #4cc9f0 0%, #3aa8d8 100%);
    }

    .bank-icon {
        background: linear-gradient(135deg, #4361ee 0%, #3f37c9 100%);
    }

    .voucher-icon {
        background: linear-gradient(135deg, #f72585 0%, #e11570 100%);
    }

    .stat-info h3 {
        font-size: 1.8rem;
        margin-bottom: 5px;
        color: var(--dark-color);
    }

    .stat-info p {
        color: #666;
        font-size: 0.9rem;
    }

    @media (max-width: 768px) {
        .row {
            flex-direction: column;
        }
        
        .col {
            width: 100%;
            margin-bottom: 15px;
        }
        
        .header {
            flex-direction: column;
            text-align: center;
            gap: 15px;
        }
        
        .tabs {
            flex-direction: column;
        }
        
        .tab {
            border-right: none;
            border-bottom: 1px solid #eee;
        }
    }

    .toggle-switch {
        position: relative;
        display: inline-block;
        width: 60px;
        height: 30px;
    }

    .toggle-switch input {
        opacity: 0;
        width: 0;
        height: 0;
    }

    .toggle-slider {
        position: absolute;
        cursor: pointer;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background-color: #ccc;
        transition: .4s;
        border-radius: 34px;
    }

    .toggle-slider:before {
        position: absolute;
        content: "";
        height: 22px;
        width: 22px;
        left: 4px;
        bottom: 4px;
        background-color: white;
        transition: .4s;
        border-radius: 50%;
    }

    input:checked + .toggle-slider {
        background-color: var(--success-color);
    }

    input:checked + .toggle-slider:before {
        transform: translateX(30px);
    }

    .form-footer {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-top: 20px;
        padding-top: 20px;
        border-top: 1px solid #eee;
    }

    .checkbox-group {
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .badge {
        display: inline-block;
        padding: 5px 10px;
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 600;
    }

    .badge-receipt {
        background: #e8f7e8;
        color: #2d572c;
    }

    .badge-payment {
        background: #fdecea;
        color: #721c24;
    }
</style>
</head>
<body>
<div class="container">
    <div class="header">
        <h1><i class="fas fa-money-bill-alt"></i> نظام إدارة الصناديق والبنوك وسندات القبض والصرف</h1>
        <div style="font-size: 0.9rem; opacity: 0.9;">
            <i class="fas fa-calendar-alt"></i> <?php echo date('Y-m-d'); ?>
        </div>
    </div>

    <?php if($message): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?=htmlspecialchars($message)?>
        </div>
    <?php endif; ?>

    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon cash-icon">
                <i class="fas fa-cash-register"></i>
            </div>
            <div class="stat-info">
                <?php
                $total_cash = $mysqli->query("SELECT SUM(current_balance) as total FROM cash_boxes WHERE is_active=1");
                $cash_total = $total_cash->fetch_assoc()['total'] ?? 0;
                ?>
                <h3><?=number_format($cash_total, 2)?> ¥</h3>
                <p>إجمالي أرصدة الصناديق</p>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon bank-icon">
                <i class="fas fa-university"></i>
            </div>
            <div class="stat-info">
                <?php
                $total_bank = $mysqli->query("SELECT SUM(current_balance) as total FROM bank_accounts WHERE is_active=1");
                $bank_total = $total_bank->fetch_assoc()['total'] ?? 0;
                ?>
                <h3><?=number_format($bank_total, 2)?> ¥</h3>
                <p>إجمالي أرصدة البنوك</p>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon voucher-icon">
                <i class="fas fa-file-invoice-dollar"></i>
            </div>
            <div class="stat-info">
                <?php
                $today_vouchers = $mysqli->query("SELECT COUNT(*) as count FROM vouchers WHERE DATE(created_at) = CURDATE()");
                $voucher_count = $today_vouchers->fetch_assoc()['count'] ?? 0;
                ?>
                <h3><?=$voucher_count?></h3>
                <p>سندات اليوم</p>
            </div>
        </div>
    </div>

    <div class="tabs">
        <div class="tab active" data-target="cash-tab"><i class="fas fa-cash-register"></i> الصناديق النقدية</div>
        <div class="tab" data-target="bank-tab"><i class="fas fa-university"></i> الحسابات البنكية</div>
        <div class="tab" data-target="voucher-tab"><i class="fas fa-file-invoice-dollar"></i> سندات القبض/الصرف</div>
        <div class="tab" data-target="view-tab"><i class="fas fa-list"></i> القوائم والسجلات</div>
    </div>

    <!-- تبويب الصناديق النقدية -->
    <div id="cash-tab" class="tab-content active">
        <div class="row">
            <div class="col">
                <div class="card">
                    <h3 class="card-title"><i class="fas fa-plus-circle"></i> إضافة صندوق نقدي جديد</h3>
                    <form method="post" id="cashForm">
                        <input type="hidden" name="action" value="create_cash_box" />
                        <div class="form-group">
                            <label class="form-label">اسم الصندوق</label>
                            <input type="text" name="name" class="form-control" required placeholder="أدخل اسم الصندوق" />
                        </div>
                        <div class="row">
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">الرصيد الافتتاحي</label>
                                    <input type="number" name="opening_balance" class="form-control" step="0.01" value="0" required />
                                </div>
                            </div>
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">حالة الصندوق</label>
                                    <div class="checkbox-group">
                                        <label class="toggle-switch">
                                            <input type="checkbox" name="is_active" checked>
                                            <span class="toggle-slider"></span>
                                        </label>
                                        <span>مفعل</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="form-group">
                            <label class="form-label">وصف (اختياري)</label>
                            <textarea name="description" class="form-control" rows="3" placeholder="أدخل وصفاً للصندوق..."></textarea>
                        </div>
                        <div class="form-footer">
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-save"></i> حفظ الصندوق
                            </button>
                            <button type="reset" class="btn">
                                <i class="fas fa-redo"></i> إعادة تعيين
                            </button>
                        </div>
                    </form>
                </div>
            </div>
            
            <div class="col">
                <div class="card">
                    <h3 class="card-title"><i class="fas fa-edit"></i> تحديث بيانات صندوق</h3>
                    <form method="post">
                        <input type="hidden" name="action" value="update_cash_box" />
                        <div class="form-group">
                            <label class="form-label">اختر الصندوق</label>
                            <select name="id" class="form-control" required>
                                <option value="">-- اختر صندوقًا --</option>
                                <?php
                                $cash_boxes_list = $mysqli->query("SELECT * FROM cash_boxes ORDER BY name");
                                while($cb = $cash_boxes_list->fetch_assoc()): ?>
                                    <option value="<?=$cb['id']?>"><?=htmlspecialchars($cb['name'])?> (<?=number_format($cb['current_balance'],2)?>)</option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">اسم الصندوق الجديد</label>
                            <input type="text" name="name" class="form-control" required />
                        </div>
                        <div class="form-group">
                            <label class="form-label">وصف (اختياري)</label>
                            <textarea name="description" class="form-control" rows="2"></textarea>
                        </div>
                        <div class="form-footer">
                            <button type="submit" class="btn">
                                <i class="fas fa-edit"></i> تحديث البيانات
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="card">
            <h3 class="card-title"><i class="fas fa-list"></i> قائمة الصناديق النقدية</h3>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>الاسم</th>
                            <th>الرصيد الحالي</th>
                            <th>الرصيد الافتتاحي</th>
                            <th>الحالة</th>
                            <th>تاريخ الإنشاء</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $cash_boxes = $mysqli->query("SELECT * FROM cash_boxes ORDER BY id DESC");
                        while($cb = $cash_boxes->fetch_assoc()): ?>
                        <tr>
                            <td><strong><?=htmlspecialchars($cb['name'])?></strong><br><small class="small"><?=htmlspecialchars($cb['description'])?></small></td>
                            <td class="amount-positive"><?=number_format($cb['current_balance'],2)?> ¥</td>
                            <td><?=number_format($cb['opening_balance'],2)?> ¥</td>
                            <td>
                                <?php if($cb['is_active']): ?>
                                    <span class="status-active"><i class="fas fa-check-circle"></i> مفعل</span>
                                <?php else: ?>
                                    <span class="status-inactive"><i class="fas fa-times-circle"></i> غير مفعل</span>
                                <?php endif; ?>
                            </td>
                            <td><?=date('Y-m-d', strtotime($cb['created_at']))?></td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- تبويب الحسابات البنكية -->
    <div id="bank-tab" class="tab-content">
        <div class="row">
            <div class="col">
                <div class="card">
                    <h3 class="card-title"><i class="fas fa-plus-circle"></i> إضافة حساب بنكي جديد</h3>
                    <form method="post">
                        <input type="hidden" name="action" value="create_bank_account" />
                        <div class="form-group">
                            <label class="form-label">اسم الحساب</label>
                            <input type="text" name="name" class="form-control" required placeholder="أدخل اسم الحساب" />
                        </div>
                        <div class="row">
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">اسم البنك</label>
                                    <input type="text" name="bank_name" class="form-control" required />
                                </div>
                            </div>
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">رقم الحساب</label>
                                    <input type="text" name="account_number" class="form-control" />
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">الفرع</label>
                                    <input type="text" name="branch_name" class="form-control" />
                                </div>
                            </div>
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">IBAN</label>
                                    <input type="text" name="iban" class="form-control" />
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">SWIFT Code</label>
                                    <input type="text" name="swift_code" class="form-control" />
                                </div>
                            </div>
                            <div class="col">
                                <div class="form-group">
                                    <label class="form-label">الرصيد الافتتاحي</label>
                                    <input type="number" name="opening_balance" class="form-control" step="0.01" value="0" required />
                                </div>
                            </div>
                        </div>
                        <div class="form-footer">
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-save"></i> حفظ الحساب البنكي
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <div class="card">
            <h3 class="card-title"><i class="fas fa-list"></i> قائمة الحسابات البنكية</h3>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>اسم الحساب</th>
                            <th>البنك</th>
                            <th>رقم الحساب</th>
                            <th>الرصيد الحالي</th>
                            <th>IBAN</th>
                            <th>الحالة</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $bank_accounts = $mysqli->query("SELECT * FROM bank_accounts ORDER BY id DESC");
                        while($ba = $bank_accounts->fetch_assoc()): ?>
                        <tr>
                            <td><strong><?=htmlspecialchars($ba['name'])?></strong><br><small class="small"><?=htmlspecialchars($ba['branch_name'])?></small></td>
                            <td><?=htmlspecialchars($ba['bank_name'])?></td>
                            <td><?=htmlspecialchars($ba['account_number'])?></td>
                            <td class="amount-positive"><?=number_format($ba['current_balance'],2)?> ¥</td>
                            <td><code><?=htmlspecialchars($ba['iban'])?></code></td>
                            <td>
                                <?php if($ba['is_active']): ?>
                                    <span class="status-active"><i class="fas fa-check-circle"></i> مفعل</span>
                                <?php else: ?>
                                    <span class="status-inactive"><i class="fas fa-times-circle"></i> غير مفعل</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- تبويب سندات القبض/الصرف -->
    <div id="voucher-tab" class="tab-content">
        <div class="card">
            <h3 class="card-title"><i class="fas fa-file-invoice-dollar"></i> إنشاء سند قبض / صرف</h3>
            <form method="post" id="voucher_form">
                <input type="hidden" name="action" value="create_voucher" />
                <div class="row">
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">نوع السند</label>
                            <select name="type" class="form-control" required>
                                <option value="receipt">سند قبض</option>
                                <option value="payment">سند صرف</option>
                            </select>
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">المبلغ</label>
                            <input type="number" name="amount" class="form-control" step="0.01" required placeholder="أدخل المبلغ" />
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">العملة</label>
                            <select name="currency_id" class="form-control">
                                <?php 
                                $currencies = $mysqli->query("SELECT id, name, code, symbol FROM currencies ORDER BY id");
                                while($c = $currencies->fetch_assoc()): ?>
                                    <option value="<?=$c['id']?>"><?=htmlspecialchars($c['name'].' ('.$c['code'].')')?></option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">سعر الصرف</label>
                            <input type="text" name="exchange_rate" class="form-control" value="1.00" />
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">الطرف المقابل</label>
                          <select name="related_to" id="related_to" class="form-control" onchange="loadRelatedParties()" required>
  <option value="">-- اختر الطرف --</option>
  <option value="customer">عميل</option>
  <option value="company">شركة / مورد</option>
</select>


                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label id="related_label">اختر العميل</label>
          <select name="related_id" class="form-control" id="related_id" required>
  <option value="">اختر الطرف أولاً</option>
</select>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">طريقة الدفع</label>
                            <select name="payment_type" class="form-control" id="payment_type">
                                <option value="cash">نقد</option>
                                <option value="bank">بنكي</option>
                                <option value="both">نقد + بنكي</option>
                            </select>
                        </div>
                    </div>
                    <div class="col" id="cash_box_select">
                        <div class="form-group">
                            <label class="form-label">اختر صندوق</label>
                            <select name="cash_box_id" class="form-control">
                                <option value="">-- لا يوجد --</option>
                                <?php 
                                $cash_boxes = $mysqli->query("SELECT * FROM cash_boxes WHERE is_active=1 ORDER BY name");
                                while($cb = $cash_boxes->fetch_assoc()): ?>
                                    <option value="<?=$cb['id']?>"><?=htmlspecialchars($cb['name'].' - ('.number_format($cb['current_balance'],2).')')?></option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                    <div class="col" id="bank_account_select" style="display:none">
                        <div class="form-group">
                            <label class="form-label">اختر الحساب البنكي</label>
                            <select name="bank_account_id" class="form-control">
                                <option value="">-- لا يوجد --</option>
                                <?php 
                                $bank_accounts = $mysqli->query("SELECT * FROM bank_accounts WHERE is_active=1 ORDER BY name");
                                while($ba = $bank_accounts->fetch_assoc()): ?>
                                    <option value="<?=$ba['id']?>"><?=htmlspecialchars($ba['name'].' - '.$ba['bank_name'].' ('.number_format($ba['current_balance'],2).')')?></option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="row" id="both_row" style="display:none">
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">المبلغ نقدًا</label>
                            <input type="number" name="cash_amount" class="form-control" step="0.01" value="0" />
                        </div>
                    </div>
                    <div class="col">
                        <div class="form-group">
                            <label class="form-label">المبلغ بنكي (سيحسب تلقائياً)</label>
                            <input type="number" id="bank_calc" class="form-control" disabled />
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">ملاحظات / وصف</label>
                    <textarea name="description" class="form-control" rows="3" placeholder="أدخل وصفاً للسند..."></textarea>
                </div>
                
                <div class="form-footer">
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-save"></i> حفظ السند
                    </button>
                    <button type="reset" class="btn">
                        <i class="fas fa-redo"></i> إعادة تعيين
                    </button>
                </div>
            </form>
        </div>
        
        <div class="card">
            <h3 class="card-title"><i class="fas fa-history"></i> أحدث السندات</h3>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>رقم السند</th>
                            <th>النوع</th>
                            <th>المبلغ</th>
                            <th>الطرف المقابل</th>
                            <th>طريقة الدفع</th>
                            <th>التاريخ</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while($v = $recent_vouchers->fetch_assoc()): ?>
                        <tr>
                            <td><strong><?=htmlspecialchars($v['voucher_number'])?></strong></td>
                            <td>
                                <?php if($v['type'] === 'receipt'): ?>
                                    <span class="badge badge-receipt"><i class="fas fa-download"></i> قبض</span>
                                <?php else: ?>
                                    <span class="badge badge-payment"><i class="fas fa-upload"></i> صرف</span>
                                <?php endif; ?>
                            </td>
                            <td class="amount-positive"><?=number_format($v['amount'],2)?> ¥</td>
                            <td><?=htmlspecialchars($v['related_name'])?> (<?=$v['related_to'] === 'customer' ? 'عميل' : 'شركة'?>)</td>
                            <td>
                                <?php if($v['payment_type'] === 'cash'): ?>
                                    <i class="fas fa-money-bill-wave"></i> نقد
                                <?php elseif($v['payment_type'] === 'bank'): ?>
                                    <i class="fas fa-university"></i> بنكي
                                <?php else: ?>
                                    <i class="fas fa-money-bill-wave"></i> <i class="fas fa-university"></i> نقد + بنكي
                                <?php endif; ?>
                            </td>
                            <td><?=date('Y-m-d H:i', strtotime($v['created_at']))?></td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- تبويب القوائم والسجلات -->
    <div id="view-tab" class="tab-content">
        <div class="row">
            <div class="col">
                <div class="card">
                    <h3 class="card-title"><i class="fas fa-cash-register"></i> إجمالي أرصدة الصناديق</h3>
                    <div class="balance-display">
                        <?php
                        $total_cash = $mysqli->query("SELECT SUM(current_balance) as total FROM cash_boxes WHERE is_active=1");
                        $cash_total = $total_cash->fetch_assoc()['total'] ?? 0;
                        ?>
                        <span id="total-cash"><?=number_format($cash_total, 2)?></span> ¥
                    </div>
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>الصندوق</th>
                                    <th>الرصيد</th>
                                    <th>الحالة</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $cash_boxes = $mysqli->query("SELECT * FROM cash_boxes ORDER BY current_balance DESC");
                                while($cb = $cash_boxes->fetch_assoc()): ?>
                                <tr>
                                    <td><?=htmlspecialchars($cb['name'])?></td>
                                    <td class="amount-positive"><?=number_format($cb['current_balance'],2)?> ¥</td>
                                    <td>
                                        <?php if($cb['is_active']): ?>
                                            <span class="status-active"><i class="fas fa-check-circle"></i> مفعل</span>
                                        <?php else: ?>
                                            <span class="status-inactive"><i class="fas fa-times-circle"></i> غير مفعل</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            
            <div class="col">
                <div class="card">
                    <h3 class="card-title"><i class="fas fa-university"></i> إجمالي أرصدة البنوك</h3>
                    <div class="balance-display">
                        <?php
                        $total_bank = $mysqli->query("SELECT SUM(current_balance) as total FROM bank_accounts WHERE is_active=1");
                        $bank_total = $total_bank->fetch_assoc()['total'] ?? 0;
                        ?>
                        <span id="total-bank"><?=number_format($bank_total, 2)?></span> ¥
                    </div>
                    <div class="table-container">
                        <table>
                            <thead>
                                <tr>
                                    <th>الحساب البنكي</th>
                                    <th>البنك</th>
                                    <th>الرصيد</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $bank_accounts = $mysqli->query("SELECT * FROM bank_accounts ORDER BY current_balance DESC");
                                while($ba = $bank_accounts->fetch_assoc()): ?>
                                <tr>
                                    <td><?=htmlspecialchars($ba['name'])?></td>
                                    <td><?=htmlspecialchars($ba['bank_name'])?></td>
                                    <td class="amount-positive"><?=number_format($ba['current_balance'],2)?> ¥</td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="card">
            <h3 class="card-title"><i class="fas fa-chart-line"></i> الإحصائيات</h3>
            <div class="row">
                <div class="col">
                    <div class="form-group">
                        <label class="form-label">إجمالي الصناديق النقدية</label>
                        <input type="text" class="form-control" value="<?=number_format($cash_total, 2)?> ¥" readonly style="font-weight: bold; color: var(--success-color);">
                    </div>
                </div>
                <div class="col">
                    <div class="form-group">
                        <label class="form-label">إجمالي الحسابات البنكية</label>
                        <input type="text" class="form-control" value="<?=number_format($bank_total, 2)?> ¥" readonly style="font-weight: bold; color: var(--primary-color);">
                    </div>
                </div>
                <div class="col">
                    <div class="form-group">
                        <label class="form-label">إجمالي السيولة</label>
                        <input type="text" class="form-control" value="<?=number_format($cash_total + $bank_total, 2)?> ¥" readonly style="font-weight: bold; color: var(--secondary-color);">
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    // إدارة التبويبات
    document.querySelectorAll('.tab').forEach(tab => {
        tab.addEventListener('click', function() {
            // إزالة النشاط من جميع التبويبات
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
            
            // إضافة النشاط للتبويب المحدد
            this.classList.add('active');
            const targetId = this.getAttribute('data-target');
            document.getElementById(targetId).classList.add('active');
        });
    });

   const relatedTo = document.getElementById('related_to');
const relatedLabel = document.getElementById('related_label');
const relatedSelect = document.getElementById('related_id');

relatedTo.addEventListener('change', ()=> {
    const v = relatedTo.value;

    relatedLabel.innerText = v === 'customer' 
        ? 'اختر العميل' 
        : 'اختر الشركة / المورد';

    relatedSelect.innerHTML = '<option value="">جاري التحميل...</option>';

    if(v === ''){
        relatedSelect.innerHTML = '<option value="">اختر الطرف أولاً</option>';
        return;
    }

    fetch("get_related_parties.php?type=" + v)
        .then(res => res.text())
        .then(data => {
            relatedSelect.innerHTML = data;
        })
        .catch(err => {
            console.error(err);
            relatedSelect.innerHTML = '<option value="">خطأ في التحميل</option>';
        });
});


    // تفاعلية طريقة الدفع
    const paymentType = document.getElementById('payment_type');
    const cashSelect = document.getElementById('cash_box_select');
    const bankSelect = document.getElementById('bank_account_select');
    const bothRow = document.getElementById('both_row');
    const voucherForm = document.getElementById('voucher_form');

    paymentType.addEventListener('change', () => {
        const v = paymentType.value;
        cashSelect.style.display = (v === 'cash' || v === 'both') ? 'block' : 'none';
        bankSelect.style.display = (v === 'bank' || v === 'both') ? 'block' : 'none';
        bothRow.style.display = (v === 'both') ? 'flex' : 'none';
    });

    // حساب المبلغ البنكي تلقائياً
    voucherForm.addEventListener('input', () => {
        const amount = parseFloat(voucherForm.amount.value || 0);
        const cashAmt = parseFloat(voucherForm.cash_amount ? voucherForm.cash_amount.value || 0 : 0);
        const bankCalc = document.getElementById('bank_calc');
        if(bankCalc) { 
            const bankAmount = amount - cashAmt;
            bankCalc.value = bankAmount >= 0 ? bankAmount.toFixed(2) : '0.00';
            
            // تحديث قيمة المبلغ البنكي في حال كان سالباً
            if(bankAmount < 0) {
                alert('المبلغ النقدي لا يمكن أن يكون أكبر من المبلغ الكلي');
                voucherForm.cash_amount.value = amount.toFixed(2);
                bankCalc.value = '0.00';
            }
        }
    });

    // التأكيد قبل الإرسال
    document.querySelectorAll('form').forEach(form => {
        form.addEventListener('submit', function(e) {
            const submitBtn = this.querySelector('button[type="submit"]');
            if(submitBtn) {
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الحفظ...';
                submitBtn.disabled = true;
            }
        });
    });

    // تأثيرات إضافية
    document.addEventListener('DOMContentLoaded', function() {
        // إضافة تأثير للبطاقات عند التمرير
        const cards = document.querySelectorAll('.card');
        cards.forEach(card => {
            card.style.opacity = '0';
            card.style.transform = 'translateY(20px)';
        });
        
        // تأثير الظهور التدريجي
        setTimeout(() => {
            cards.forEach((card, index) => {
                setTimeout(() => {
                    card.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        }, 300);
        
        // تحديث الوقت في الهيدر
        function updateTime() {
            const now = new Date();
            const timeStr = now.toLocaleTimeString('ar-SA');
            const dateStr = now.toLocaleDateString('ar-SA');
            const timeElement = document.querySelector('.header div');
            if(timeElement) {
                timeElement.innerHTML = `<i class="fas fa-calendar-alt"></i> ${dateStr} <i class="fas fa-clock" style="margin-right:10px"></i> ${timeStr}`;
            }
        }
        
        updateTime();
        setInterval(updateTime, 1000);
    });

    // رسالة تأكيد عند محاولة ترك الصفحة
    window.addEventListener('beforeunload', function (e) {
        const forms = document.querySelectorAll('form');
        let hasUnsavedChanges = false;
        
        forms.forEach(form => {
            if(form.checkValidity && !form.checkValidity()) {
                hasUnsavedChanges = true;
            }
        });
        
        if(hasUnsavedChanges) {
            e.preventDefault();
            e.returnValue = 'لديك تغييرات غير محفوظة. هل تريد بالتأكيد مغادرة الصفحة؟';
        }
    });
</script>
</body>
</html>